# -*- coding: utf-8 -*-
"""
===============================================================================
file            : whsk_F3.py
===============================================================================

This Python script derives "Waldhöhenstrukturkarte" (WHSK) from normalised 
digital surface models (nDSM) in tif-format. For deriving WHSK height values 
are aggregated to a pixel size of 5m (original raster resolution = 1m) keeping 
the highest value. Subsequently the data is classified into height classes 
(class width = 1m).

It was devised within the scope of the "F3 - Flächendeckende 
Fernerkundungsbasierte Forstliche Strukturdaten" (F3- Area-wide remote sensing 
based forest structural data) project by project partners Forest Research 
Institute of Baden-Württemberg (Forstliche Versuchs- und Forschungsanstalt 
Baden-Württemberg - FVA) and Northwest German Forest Research Institute 
(Nordwestdeutsche Forstliche Versuchsanstalt - NW-FVA).
For further information go to www.waldwissen.net/technik/inventur/f3/ or contact
Petra Adler, Petra.Adler@forst.bwl.de (FVA)
Jörg Ackermann, Joerg.Ackermann@nw-fva.de (NW-FVA)

This script is published under GNU General Public License Version 3, 29 June 2007.

"""

# ==============================================================================
# Import moduls
# ==============================================================================

import os
import re
from glob import glob
import multiprocessing
from shutil import rmtree
import arcpy
import math

# ==============================================================================
# Environment settings
# ==============================================================================

# Arcpy Env
arcpy.env.overwriteOutput = True

color_map_folder = "color_maps"
color_map_file = "colormap_whsk_2.clr"

# ==============================================================================
# Define functions
# ==============================================================================

def create_folders(out_folder, metric, pixel_sizes):
    """If not already exists, create folders with the name of the metric and the pixel size."""""
    for pixel in pixel_sizes:
        folder = r"{}\{}_{}".format(out_folder, metric, pixel)
        if not os.path.isdir(folder):
            os.mkdir(folder)
            print(r"Directory {}\{}_{} created.".format(out_folder, metric, pixel))

def remap(start, width, end):
    """Create RemapRange for arcpy.gp.Reclassify_sa(). Input: start = lowest value for reclassification;
     width = width of classes; end = highest value for reclassification"""
    num_classes = int(math.ceil((end-start)/(width*1.0)))
    classes = []
    for n in range(num_classes):
        lower = (n * width) + start
        upper = lower + width
        classes.append("{} {} {}".format(lower, upper, upper))
    return ';'.join(classes)

def worker():

    print("...Resampling nDSM to 5 m resolution...")
    in_dir = tif_ndsm
    file_list = glob(os.path.join(in_dir, '*.tif'))  # mit pfad
    file_list_2 = [f for f in os.listdir(in_dir) if f.endswith('.tif')]  # ohne pfad

    create_folders(output_folder, 'temp_whsk', pixel_step)

    file_list_2_length = len(file_list_2)
    arcpy.CheckOutExtension("Spatial")
    for n in xrange(file_list_2_length):
        arcpy.gp.Aggregate_sa(file_list[n], os.path.join(temp_folder, file_list_2[n]),
                              pixel_step, "MAXIMUM", "TRUNCATE", "DATA")

    print("...Generating height classes...")
    in_dir = os.path.join(output_folder, "temp_whsk_{}".format(pixel_step))
    file_list = glob(os.path.join(in_dir, '*.tif'))
    file_list_2 = [f for f in os.listdir(in_dir) if f.endswith('.tif')]
    output_name_list = []

    for filename in file_list_2:  # define name of output files
        print(re.sub("\D", "", filename))
        output_name = "whsk{}_".format(pixel_step) + re.sub("\D", "", filename) + ".tif"
        output_name_list.append(output_name)

    create_folders(output_folder, 'whsk', pixel_step)

    output_name_list_length = len(output_name_list)
    arcpy.CheckOutExtension("Spatial")
    for n in xrange(output_name_list_length):
        arcpy.gp.Reclassify_sa(file_list[n], "Value", remap_classes,
                               os.path.join(output_folder, "whsk_{}".format(pixel_step), output_name_list[n]), "DATA")

    # *** Add color ramp ***

    # list files to which we want to apply the color map
    for root, dirs, files in os.walk(os.path.abspath(output_folder)):
        dirList = [os.path.join(root, file) for file in files if file.endswith(".tif")]

    # Path to the directory where the script is saved
    dir_script = os.path.dirname(os.path.realpath(__file__))

    # In this directory, search the directory with the color maps
    for root, subdirs, files in os.walk(dir_script):
        for d in subdirs:
            if d == color_map_folder:
                dir_color_maps = os.path.join(root, d)

    # Search the color map for this script
    try:
        dir_color_maps
        for root, dirs, files in os.walk(dir_color_maps):
            for f in files:
                if f == color_map_file:
                    color_map = os.path.join(root, f)
    except NameError:
        print("Color map folder not found")
        pass

    try:
        color_map
        for file in dirList:
            arcpy.AddColormap_management(in_raster=file, in_template_raster="",
                                         input_CLR_file=color_map)
        print("Color map added to the rasters")
    except NameError:
        print("Color map file not found")
        pass

    for filename in os.listdir(temp_folder):
        arcpy.Delete_management(os.path.join(temp_folder, filename))

    try:
        rmtree(temp_folder)
    except WindowsError as e:
        print(e)

# ==============================================================================
# Start processing
# ==============================================================================

if __name__ == "__main__":

    # ==============================================================================
    # Start: Define parameters and set paths
    # ==============================================================================
    # Parameters
    cores = multiprocessing.cpu_count() - 1
    pixel_step = "5"  # geometric resolution of output (WHSK)
    remap_classes = remap(-1, 1, 55)  # creating "RemapRange" for Reclassify_sa()

    # Paths
    # set path to input data: nDSM in tif-format
    tif_ndsm = r"D:\F3\forest_structure\nDSM"
    # set path to output folder
    output_folder = r"D:\F3\forest_structure\WHSK"
    # temp path (is created automatically)
    temp_folder = os.path.join(output_folder, r"temp_whsk_{}".format(pixel_step))

    print("Paths and parameters set")
    # ==============================================================================
    # End: Set paths and define parameters
    # ==============================================================================

    letsgo = worker()