# -*- coding: utf-8 -*-
"""
===============================================================================
file            : waldtyp_F3.py
===============================================================================
This Python-script derives areas of open forest, dense forest and gaps. The 
result is a raster (tif-format) that contains these three classes. Also, a canopy 
cover map is created.

It was devised within the scope of the "F3 - Flächendeckende 
Fernerkundungsbasierte Forstliche Strukturdaten" (F3- Area-wide remote sensing 
based forest structural data) project by project partners Forest Research 
Institute of Baden-Württemberg (Forstliche Versuchs- und Forschungsanstalt 
Baden-Württemberg - FVA) and Northwest German Forest Research Institute 
(Nordwestdeutsche Forstliche Versuchsanstalt - NW-FVA).
For further information go to www.waldwissen.net/technik/inventur/f3/ or contact
Petra Adler, Petra.Adler@forst.bwl.de (FVA)
Jörg Ackermann, Joerg.Ackermann@nw-fva.de (NW-FVA)

This script is published under GNU General Public License Version 3, 29 June 2007.

"""

# ==============================================================================
# Import moduls
# ==============================================================================

import os
import multiprocessing
import arcview 
import arcpy
from arcpy.sa import *
from arcpy import env
from shutil import rmtree
from glob import glob

# ==============================================================================
# Environment settings
# ==============================================================================

arcpy.env.overwriteOutput = True
arcpy.env.pyramid = "NONE"
arcpy.CheckOutExtension("spatial")

# ==============================================================================
# Start: Set paths and define parameters
# ==============================================================================

# Paths
# Path to input directory: nDSM im .LAZ Format
cloud_ndsm = r"D:\F3\forest_structure\nDSM_laz"
# Path to output directory
output_folder = r"D:\F3\forest_structure"
# Path to "bin" folder of LAStools installation, e.g. C:\LAStools\bin
lastools = r"S:\software\LAStools19\bin"
lasgrid = lastools + r"\lasgrid.exe"

# Parameters
height = 3
aggregate_cc = 25
cores = multiprocessing.cpu_count() - 1

color_map_folder = "color_maps"
color_map_file = "colormap_gap.clr"

# Define folders
folder_name_list = ["0_tif_files", "1_reclass", "2_focal", "3_reclass", "4_group", "5_5000", "6_reclass",
                    "7_nibble", "8_reclass", "9_rastercalc", "10_reclass", "11_group", "12_10",
                    "13_reclass", "14_nibble", "waldtyp4copy"]

output_folder_list = ["waldtyp", "ueberschirmung"]

# ==============================================================================
# Define functions
# ==============================================================================

def create_folder(output_f, folder_name):
    if not os.path.isdir(os.path.join(output_f, folder_name)):
        os.mkdir(os.path.join(output_f, folder_name))


def create_lof(input_dir):
    """ Create and return a list of files as .txt from the files in the input directory"""
    dirList = []
    for root, dirs, files in os.walk(os.path.abspath(input_dir)):
        for pointcloud in files:
            dirList.append(os.path.join(root, pointcloud))
    with open('files.txt', 'w') as f:
        [f.write("{}\n".format(item)) for item in dirList]
    f.close()
    return r"files.txt"


# ==============================================================================
# Start processing
# ==============================================================================

create_folder(output_folder, "temp_waldtyp")
temp_folder = os.path.join(output_folder, "temp_waldtyp")

[create_folder(temp_folder, name) for name in folder_name_list]
[create_folder(output_folder, name) for name in output_folder_list]

# 0 Convert .laz to .tif
parameters = r" -step 1 -otif -elevation -highest -fill 3 -cores {} -odir {}".format(cores, os.path.join(temp_folder, "0_tif_files"))
lastools_query = lasgrid + " -lof " + create_lof(cloud_ndsm) + parameters
os.system(lastools_query)


# *** 1 reclass ***
# output is classification in "tree" (height >= 3 m; value = 1) and "no tree" (height < 3 m ; value = 0)
dirList = glob(os.path.join(temp_folder, "0_tif_files", "*.tif"))

for f in dirList:
    outReclass = Reclassify(f, "VALUE", "-2 {} 0;{} 60 1".format(height, height), "NODATA")
    outReclass.save(os.path.join(temp_folder, "1_reclass", "{}".format(f[-13:])))

# *** 2 focal mean ***
# output is tree cover (value 1 = 100%)
dirList = glob(os.path.join(temp_folder, "1_reclass", "*.tif"))

for f in dirList:
    out_raster = FocalStatistics(f, "Circle 25 CELL", "MEAN", "DATA")
    out_raster.save(os.path.join(temp_folder, "2_focal", "{}".format(f[-13:])))

# *** 3 second reclass ***
# output is classification into open (tree cover <= 0,5999; value = 0) and dense (tree cover >= 0,6; value = 1) forest
dirList = glob(os.path.join(temp_folder, "2_focal", "*.tif"))

for f in dirList:
    out_raster = Reclassify(f, "VALUE", "0 0,5999 0;0,6 1 1", "NODATA")
    out_raster.save(os.path.join(temp_folder, "3_reclass", "{}".format(f[-13:])))

# *** 4 grouping into regions ***
# output is groups of concatenated pixels belonging to the same forest type (i. e. open or dense forest)
dirList = glob(os.path.join(temp_folder, "3_reclass", "*.tif"))

for f in dirList:
    out_raster = RegionGroup(f, "EIGHT", "WITHIN", "ADD_LINK")
    out_raster.save(os.path.join(temp_folder, "4_group", "{}".format(f[-13:])))

# *** 5 nach attributen extahieren > 5000 ***
# output is all regions that are >= 5000 pixels; no data value is assigned to pixels belonging to smaller regions
dirList = glob(os.path.join(temp_folder, "4_group", "*.tif"))

for f in dirList:
    out_raster = arcpy.sa.ExtractByAttributes(f, "Count >= 5000")
    out_raster.save(os.path.join(temp_folder, "5_5000","{}".format(f[-13:])))

# *** 6 third reclass ***
# output is classification of the remaining regions into open (value = 0) and dense (value = 1) forest
dirList = glob(os.path.join(temp_folder, "5_5000", "*.tif"))

for f in dirList:
    out_raster = Reclassify(in_raster=f, reclass_field="LINK", remap="0 0;1 1", missing_values="NODATA")
    out_raster.save(os.path.join(temp_folder, "6_reclass", "{}".format(f[-13:])))

# *** 7 nibble ***
# Use second classification into open/dense as a mask for filling small regions (<5000 pixels)
# of the first classification into open and dense forest, these regions are filled using the nearest neighbour
# pixels value (dissolving small regions into the surrounding region)
nibble_dirList = glob(os.path.join(temp_folder, "6_reclass", "*.tif"))

nibble_dirList.sort()
dirList = glob(os.path.join(temp_folder, "3_reclass", "*.tif"))

dirList.sort()
for num, f in enumerate(dirList):
    out_raster = Nibble(f, nibble_dirList[num], "DATA_ONLY")
    out_raster.save(os.path.join(temp_folder, "7_nibble", "{}".format(f[-13:])))

# *** 8 reclassify ***
# output is reclassification of open/dense forest where open forest = 1 and dense forest = 3
dirList = glob(os.path.join(temp_folder, "7_nibble", "*.tif"))

for f in dirList:
    out_raster = Reclassify(f, "VALUE", "0 1;1 3", "NODATA")
    out_raster.save(os.path.join(temp_folder, "8_reclass", "{}".format(f[-13:])))

# *** 9  calculate raster ***
# output is combination of classification into open/dense forest and classification into tree (>3 m) and no tree,
# resulting in values
# 0 = ignore
# 1 = no trees within open forest
# 2 = trees (>3m) within dense forest
# 3 = no trees within dense forest (gaps)
dirList = glob(os.path.join(temp_folder, "8_reclass", "*.tif"))

dirList.sort()
bin_dirList = glob(os.path.join(temp_folder, "1_reclass", "*.tif"))
bin_dirList.sort()
for num, f in enumerate(dirList):
    out_raster = Raster(f) - Raster(bin_dirList[num])
    out_raster.save(os.path.join(temp_folder, "9_rastercalc", "{}".format(f[-13:])))

# *** 10 reclassify ***
# in the output classes 0 and 1 from step above (output in folder "9_rastercalc") are combined into class 1
dirList = glob(os.path.join(temp_folder, "9_rastercalc", "*.tif"))

for f in dirList:
    out_raster = Reclassify(f, "VALUE", "0 1;1 1;2 2;3 3", "NODATA")
    out_raster.save(os.path.join(temp_folder, "10_reclass", "{}".format(f[-13:])))

# 11 *** grouping into regions ***
# output is groups of concatenated pixels belonging to the same class (class value is stored as "LINK")
dirList = glob(os.path.join(temp_folder, "10_reclass", "*.tif"))

for f in dirList:
    out_raster = RegionGroup(f, "EIGHT", "WITHIN", "ADD_LINK", None)
    out_raster.save(os.path.join(temp_folder, "11_group", "{}".format(f[-13:])))

# *** 12 nach attributen extahieren < 10 ***
# output is regions with >= 10 pixels (10m2 minimum gap size), no data is assigned to smaller regions
dirList = glob(os.path.join(temp_folder, "11_group", "*.tif"))

for f in dirList:
    out_raster = arcpy.sa.ExtractByAttributes(f, "Count > 9")
    out_raster.save(os.path.join(temp_folder, "12_10", "{}".format(f[-13:])))

# *** 13 reclass ***
# output is reclassification of the regions >= 10 pixels into their original classes
dirList = glob(os.path.join(temp_folder, "12_10", "*.tif"))

for f in dirList:
    outReclass = Reclassify(f, "LINK", "1 1;1 2 2;2 3 3", "NODATA")
    outReclass.save(os.path.join(temp_folder, "13_reclass", "{}".format(f[-13:])))

# *** 14 nibble ***
# assigning pixels of small regions (<10m2) the pixel value of nearest neighbour (dissolving small regions)
nibble_dirList = glob(os.path.join(temp_folder, "13_reclass", "*.tif"))

nibble_dirList.sort()
dirList = glob(os.path.join(temp_folder, "10_reclass", "*.tif"))
dirList.sort()

for num, f in enumerate(dirList):
    out_raster = Nibble(f, nibble_dirList[num], "DATA_ONLY")
    out_raster.save(os.path.join(temp_folder, "14_nibble", "{}".format(f[-13:])))

# *** Clip rasters ***
# reduce raster extent to 1*1km for forest type and canopy cover
# Get paths to final forest type tiles
input_names = glob(os.path.join(temp_folder, "14_nibble", "*.tif"))

input_names.sort()

# Get paths to canopy cover tiles
input_names_cc = glob(os.path.join(temp_folder, "2_focal", "*.tif"))

input_names_cc.sort()
env.workspace = output_folder

spatial_reference = arcpy.Describe(input_names[0]).spatialReference
arcpy.env.outputCoordinateSystem = spatial_reference

epsg_code = 31467#spatial_reference.factoryCode
if epsg_code is 0:  # custom projections don't have factory code
    spatial_info = spatial_reference.exportToString()

 
    if 'Zone_32N' in spatial_info:
        epsg_code = 25832
    if 'Zone_33N' in spatial_info:
        epsg_code = 25833
    if 'Zone_2' in spatial_info:
        epsg_code = 31466
								
						 
    if 'Zone_4' in spatial_info:
        epsg_code = 31468

for num, f in enumerate(input_names):
    # Get 1*1km extent
    if epsg_code == 31467 or epsg_code == 31466 or epsg_code == 31468:
        xmin = f[-12:-8] + "000"
        ymin = f[-8:-4] + "000"
        file_name = f[-12:-4]
    elif epsg_code == 25832 or epsg_code == 25833:
        xmin = f[-11:-8] + "000"
        ymin = f[-8:-4] + "000"
        file_name = f[-13:-4]
    xmax = str(int(xmin) + 1000)
    ymax = str(int(ymin) + 1000)
    coord = [xmin, ymin, xmax, ymax]

    # Clip forest type and canopy cover
    arcpy.Clip_management(in_raster=f, rectangle=(" ".join(coord)),
                          out_raster=temp_folder + r"\waldtyp4copy" + r"\waldtyp_{}.tif".format(file_name))
    # use CopyRaster for ensuring each raster uses the same no data value and pixel type (pixel depth)
    in_ras = temp_folder + r"\waldtyp4copy" + r"\waldtyp_{}.tif".format(file_name) 
    out_ras = output_folder + r"\waldtyp" + r"\waldtyp_{}.tif".format(file_name)                   
    arcpy.CopyRaster_management(in_ras,out_ras,"","","0","NONE","NONE","2_BIT","NONE","NONE","TIFF","NONE")
    # create temporary files for aggregating canopy cover
    tmp_cc = Aggregate(input_names_cc[num], aggregate_cc, "MEDIAN")
    arcpy.Clip_management(tmp_cc, rectangle=(" ".join(coord)),
                          out_raster=os.path.join(output_folder, "ueberschirmung",
                          "ueberschirmung_{}.tif".format(file_name)))

# *** Add color ramp ***

# list files to which we want to apply the color map
dirList = glob(os.path.join(output_folder, "waldtyp", "*.tif"))

# Path to the directory where the script is saved
dir_script = os.path.dirname(os.path.realpath(__file__))

# In this directory, search the directory with the color maps
for root, subdirs, files in os.walk(dir_script):
    for d in subdirs:
        if d == color_map_folder:
            dir_color_maps =os.path.join(root, d)

# Search the color map for this script
try:
    dir_color_maps
    for root, dirs, files in os.walk(dir_color_maps):
        for f in files:
            if f == color_map_file:
                color_map = os.path.join(root, f)
except NameError:
    print("Color map folder not found")
    pass

try:
    color_map
    for file in dirList:
        arcpy.AddColormap_management(in_raster=file, in_template_raster="",
                                     input_CLR_file=color_map)
    print("Color map added to the rasters")
except NameError:
    print("Color map file not found")
    pass

# *** remove temporary data ***
for folder in folder_name_list:
    for filename in os.listdir(os.path.join(temp_folder, folder)):
        arcpy.Delete_management(os.path.join(temp_folder, folder, filename))

try:
    rmtree(temp_folder)
except WindowsError as e:
    print(e)
