# ===============================================================================
# file            : soilgrids_F3.R
# ===============================================================================

# This R script prepares soil data (bulk density, coarse fragments, cation exchange 
# capacity (at pH 7), nitrogen) from soilgrids.org for utilisation as predictors 
# in a timber volume and above-ground biomass modeling process.

# It was devised within the scope of the "F3 - Flchendeckende 
# Fernerkundungsbasierte Forstliche Strukturdaten" (F3- Area-wide remote sensing 
# based forest structural data) project by project partners Forest Research 
# Institute of Baden-Wrttemberg (Forstliche Versuchs- und Forschungsanstalt 
# Baden-Wrttemberg - FVA) and Northwest German Forest Research Institute 
# (Nordwestdeutsche Forstliche Versuchsanstalt - NW-FVA).
# For further information go to www.waldwissen.net/technik/inventur/f3/ or contact
# Petra Adler, Petra.Adler@forst.bwl.de (FVA)
# Jrg Ackermann, Joerg.Ackermann@nw-fva.de (NW-FVA)

# This script is published under GNU General Public License Version 3, 29 June 2007.

# =================================================================================

##############################
# Als Eingangsdaten mssen die folgenden Layer von soilgrids.org heruntergeladen werden:
# 1. Bulk density
# 2. Coarse fragments
# 3. Cation exchange capacity (at ph 7)
# 4. Nitrogen
# Die Daten mssen jeweils fr alle 6 Bodentiefen heruntergeladen werden und im selben Ordner abgelegt werden
# Die Benennung der Daten muss dem folgenden Schema folgen:
# 1. bulkdensity_1, bulkdensity_2, ... bulkdensity_6
# 2. coarse_fragments_1, coarse_fragments_2, ... coarse_fragments_6
# 3. cation_exchange_capacity_1, cation_exchange_capacity_2, ... cation_exchange_capacity_6
# 4. nitrogen_1, nitrogen_2, ... nitrogen_6
################################
# Das Skript aggregiert die Daten zu den Bodentiefeklassen:
# 1. Oberboden (0-30 cm)
# 2. Hauptwurzelraum (0-100 cm)
# 3. komplettes Profil (0-200 cm)
# Und berechnet
# (a) Stickstoffvorrat (t/ha) pro Klasse
# (b) maximale Bodendichte im Hauptwurzelraum (kg/dm) und
# (c) die potentielle Kationenaustauschkapazitt (kmol(c)/ha) pro Klasse
#################################

library(raster)
library(rgdal)

###############
# Folgende Pfadangaben mssen angepasst werden

setwd('D:/')

# Ordner, der die von Soilgrids heruntergeladenen Daten enthlt. 
    input.folder <- 'D:/F3/Daten/Zusatzdaten/Standortsdaten/Soilgrids/raw' 
# Ordner, in dem die pro Bodentiefestufe aggregierten Daten abgelegt werden sollen
    output.folder <- 'D:/F3/Daten/Zusatzdaten/Standortsdaten/Soilgrids/'        

##########################################################
# Bulk Density (Bodendichte)
##########################################################
# Berechnet die maximale Bodendichte im Hauptwurzelraum (0-100cm)
##########################################################

      flist.bulk <- list.files(input.folder, pattern = "bulk", full.names = T)
      b1 <- stack(flist.bulk[1:5])
      
      #convert bulk density to kg/dm
      b2 <- b1/100
      
      b2 <- calc(b2, max, na.rm=T)
      
      writeRaster(b2, paste(output.folder, 'bulkdensity_max.tif', sep='/'), overwrite=T)

##########################################################
# Berechnet den Stickstoffvorrat in t/ha
##########################################################

      flist.bulk <- list.files(input.folder, pattern = "bulk", full.names = T)
      b1 <- stack(flist.bulk)
      
      flist.coarse <- list.files(input.folder, pattern = "coarse", full.names = T)
      c1 <- stack(flist.coarse)
      
      flist.nitro <- list.files(input.folder, pattern = "nitro", full.names = T)
      n1 <- stack(flist.nitro)
      
      #convert bulk density to kg/dm
      b2 <- b1/100
      #convert coarse fragments to Vol%
      c2 <- c1/10
      c2 <- (100-c2)/100
      #convert nitrogen to g/kg
      n2 <- n1/100
      
      # g/m
      n3 <- b2*n2*c2*10
      n3 <- n3*c(5,10,15,30,40,100)
      
      # t/ha
      n4 <- n3/1000/1000*10000
      
      n4.30 <- calc(n4[[1:3]], sum, na.rm=T, filename=paste(output.folder, 'nitrogen1.tif', sep='/'))
      n4.100 <- calc(n4[[1:5]], sum, na.rm=T, filename=paste(output.folder, 'nitrogen2.tif', sep='/'))
      n4.200 <- calc(n4[[1:6]], sum, na.rm=T, filename=paste(output.folder, 'nitrogen3.tif', sep='/'))
      
      
##########################################################
# Kationenaustauschkapazitt
##########################################################

      flist.bulk <- list.files(input.folder, pattern = "bulk", full.names = T)
      b1 <- stack(flist.bulk)
      
      flist.coarse <- list.files(input.folder, pattern = "coarse", full.names = T)
      c1 <- stack(flist.coarse)
      
      flist.cation <- list.files(input.folder, pattern = "cation", full.names = T)
      cec1 <- stack(flist.cation)
      
      #convert bulk density to kg/dm
      b2 <- b1/100
      #convert coarse fragments to Vol%
      c2 <- c1/10
      c2 <- (100-c2)/100
      #convert cec to cmol(c)/kg
      cec2 <- cec1/10
      
      # cmol(c)/m
      cec3 <- b2*c2*cec2*10
      cec3 <- cec3*c(5,10,15,30,40,100)
      
      # kmol(c)/ha
      cec4 <- cec3/100000*10000
      
      cec4.30 <- calc(cec4[[1:3]], sum, na.rm=T, filename=paste(output.folder, 'cec1.tif', sep='/'))
      cec4.100 <- calc(cec4[[1:5]], sum, na.rm=T, filename=paste(output.folder, 'cec2.tif', sep='/'))
      cec4.200 <- calc(cec4[[1:6]], sum, na.rm=T, filename=paste(output.folder, 'cec3.tif', sep='/'))
      
###########################################################




