# -*- coding: utf-8 -*-
"""
===============================================================================
file            : rename_F3.py
===============================================================================

This Python script renames point clouds processed in sure or match-t and 
dtm files from lgln to the form dsm/dtm_32(x)(y).laz/las or dsm/dtm_(x)(y).laz/las 
depending on the input coordinate system (UTM or Gauss-Krüger). This process is 
part of DSM/nDSM creation with cloud2ndom_F3.py.

It was devised within the scope of the "F3 - Flächendeckende 
Fernerkundungsbasierte Forstliche Strukturdaten" (F3- Area-wide remote sensing 
based forest structural data) project by project partners Forest Research 
Institute of Baden-Württemberg (Forstliche Versuchs- und Forschungsanstalt 
Baden-Württemberg – FVA) and Northwest German Forest Research Institute 
(Nordwestdeutsche Forstliche Versuchsanstalt – NW-FVA).
For further information go to www.waldwissen.net/technik/inventur/f3/ or contact
Petra Adler, Petra.Adler@forst.bwl.de (FVA)
Jörg Ackermann, Joerg.Ackermann@nw-fva.de (NW-FVA)

This script is published under GNU General Public License Version 3, 29 June 2007.

"""

#==============================================================================
# Import moduls
#==============================================================================

import os
import re

#==============================================================================
# Define function
#==============================================================================

def rename_F3(dir_path, epsg):
    
    if epsg == 25832:
        index = "32"
        posdiff = 0
        dtm_pattern_lglbw = re.compile('DGM_32[0-9]{3}_[0-9]{4}\.[laszLASZ]{3}')
    elif epsg == 25833:
        index = "33"
        posdiff = 0
    elif epsg in (31466, 31467, 31468):
        index = ""
        posdiff = 1
        dtm_pattern_lglbw = re.compile('DGM_[0-9]{4}_[0-9]{4}\.[laszLASZ]{3}')
    
    dtm_pattern_lgln = re.compile('[0-9]{8}\.[laszLASZ]{3}')
    
    matcht_pattern = re.compile('[DSMdsm]{3}_[Cloudc]{5}_[0-9]{7}_[0-9]{6}\.[laszLASZ]{3}')
    matcht_pattern_old = re.compile('DSM_Cloud_[0-9]{1}_[0-9]{7}_[0-9]{6}\.[laszLASZ]{3}')

    sure_pattern = re.compile('DSM_Cloud_[0-9]{4}_[0-9]{4}\.[laszLASZ]{3}')
    
    for file in os.listdir(dir_path):
        if matcht_pattern.match(file):
            file_new = "DSM_" + index + file[18:21] + file[10:14] + file[-4:]

        elif matcht_pattern_old.match(file):
            file_new = "DSM_" + index + file[20:23] + file[12:16] + file[26:30]

        elif sure_pattern.match(file):
            file_new = "DSM_" + index + file[11-posdiff:14] + file[15:len(file)]

        elif dtm_pattern_lglbw.match(file):
            file_new = "DTM" + file[3:9 - posdiff] + file[10 - posdiff:len(file)]

        elif dtm_pattern_lgln.match(file):
            file_new = "DTM_3" + file[0:len(file)]

        else:
            print("No match in {}  found, no rename".format(file))
            continue

        os.rename(os.path.join(dir_path, file), os.path.join(dir_path, file_new))
        print("file {} renamed to {}".format(file, file_new))