# -*- coding: utf-8 -*-
'''
===============================================================================
file            : rauigkeit_F3.py
===============================================================================
This Python-script calculates indicators for canopy roughness from normalised 
digital surface models (nDSM) in point cloud (laz)-format. Roughness is 
indicated by two different values: standard deviation in height and height 
distance between the 5th and 95th percentile. These values are each calculated 
for output resolutions of 20m, 50m and 100m. Output are rasters in tif-format.

It was devised within the scope of the "F3 - Flächendeckende 
Fernerkundungsbasierte Forstliche Strukturdaten" (F3- Area-wide remote sensing 
based forest structural data) project by project partners Forest Research 
Institute of Baden-Württemberg (Forstliche Versuchs- und Forschungsanstalt 
Baden-Württemberg - FVA) and Northwest German Forest Research Institute 
(Nordwestdeutsche Forstliche Versuchsanstalt - NW-FVA).
For further information go to www.waldwissen.net/technik/inventur/f3/ or contact
Petra Adler, Petra.Adler@forst.bwl.de (FVA)
Jörg Ackermann, Joerg.Ackermann@nw-fva.de (NW-FVA)

This script is published under GNU General Public License Version 3, 29 June 2007.

'''

# ==============================================================================
# Import moduls
# ==============================================================================

import os
import glob
import multiprocessing
from shutil import rmtree
import arcpy


# ==============================================================================
# Define functions
# ==============================================================================

def create_folders(out_folder, metric, pixel_sizes):
    """If not already exists, create folders with the name of the metric and the pixel size."""""
    for pixel in pixel_sizes:
        folder = r"{}\{}_{}".format(out_folder, metric, pixel)
        if not os.path.isdir(folder):
            os.mkdir(folder)
            print("Directory {}\{}_{} created.".format(out_folder, metric, pixel))


def create_lof(input_dir):
    """ Create and return a list of files as .txt from the files in the input directory"""
    dirList = [[os.path.join(root, pointcloud) for pointcloud in files] for root, dirs, files in
               os.walk(os.path.abspath(input_dir))]
    with open('files.txt', 'w') as f:
        [f.write("{}\n".format(item)) for item in dirList[0]]
    f.close()
    return r"files.txt"


def remove_folders(out_f, metric, pixel_sizes):
    """If folder exists, delete it."""""
    for pixel in pixel_sizes:
        folder = r"{}\{}_{}".format(out_f, metric, pixel)
        if os.path.isdir(folder):
            rmtree(folder)
            print("Directory {}\{}_{} removed.".format(out_f, metric, pixel))


def worker():
    #     # Calculate standard deviation of heights
    output_rau = os.path.join(output_folder, "rauigkeit")
    if not os.path.isdir(output_rau):
        os.mkdir(output_rau)
        print("{} created".format(output_rau))

    create_folders(output_rau, 'std', pixel_step)
    for n in pixel_step:
        parameters = " -drop_withheld -step {} -epsg {} -std -cores {} " \
                     "-otif -odir {}\std_{}".format(n, epsg_code, cores, output_rau, n)
        lastools_query = lasgrid + " -lof " + create_lof(cloud_dsm) + parameters
        os.system(lastools_query)

        # change prefix of files names
        current_output_dir = os.path.join(output_rau, "std_" + str(n))
        for fn in os.listdir(current_output_dir):
            fn_new = "rauigkeit_std" + str(n) + "_" + fn[4:]
            os.rename(os.path.join(current_output_dir, fn), os.path.join(current_output_dir, fn_new))

    # Calculate percentiles (5th and 95th) of heights
    create_folders(output_rau, 'perc', pixel_step)
    for n in pixel_step:
        parameters = " -height_cutoff -1 -drop_withheld -step {} -epsg {} -p 5 95 -cores {} " \
                     "-olaz -odir {}\perc_{} -odix _{}".format(n, epsg_code, cores, output_rau, n, n)
        lastools_query = lascanopy + " -lof " + create_lof(cloud_dsm) + parameters
        os.system(lastools_query)

    # Calculate difference between p5 and p95
    create_folders(output_rau, 'diff', pixel_step)
    for n in pixel_step:
        files = glob.glob(r'{}\perc_{}\*'.format(output_rau, n))
        files.sort()
        max_len = len(files) - 1
        # Generate matching pairs of tiles (same extent, one p5, the other p95) as input for lasdiff.exe
        input_lasdiff = [("-i {} -i {}".format(files[i + 1], files[i])) for i in range(0, max_len, 2)]

        for i in range(0, len(input_lasdiff)):
            parameters = r" {} -odir {}\diff_{} -ocut 3 -odix p95_p05_diff -olaz".format(input_lasdiff[i],
                                                                                         output_rau, n)
            lastools_query = lasdiff + parameters
            os.system(lastools_query)

    # Create output raster in tif-format
    create_folders(output_rau, 'perz', pixel_step)
    for n in pixel_step:
        parameters = r" -step {} -otif -epsg {} -cores {} -ocut 13 -odix _p95p05_diff -odir {}\perz_{}".format(n,
                                                                                                               epsg_code,
                                                                                                               cores,
                                                                                                               output_rau,
                                                                                                               n)
        lastools_query = lasgrid + " -lof " + create_lof("{}\diff_{}".format(output_rau, n)) + parameters
        os.system(lastools_query)

        # change prefix of files names
        current_output_dir = os.path.join(output_rau, "perz_" + str(n))
        for fn in os.listdir(current_output_dir):
            digits_count = len(str(n))
            fn_new = "rauigkeit_perz" + str(n) + "_" + fn[4:-17-digits_count] + fn[-4:]
            os.rename(os.path.join(current_output_dir, fn), os.path.join(current_output_dir, fn_new))

    # Delete intermediate data
    remove_folders(output_rau, 'perc', pixel_step)
    remove_folders(output_rau, 'diff', pixel_step)
    os.remove(r'files.txt')

    # Add symbology to rasters by using a template raster
    symbology_folder = "color_maps"
    symbolgy_layer_std = "template_rauigkeit_std.tif.lyr"
    symbolgy_layer_perz = "template_rauigkeit_perz.tif.lyr"

    dir_script = os.path.dirname(os.path.realpath(__file__))  # the directory where the script is
    # In this directory, search the directory with the lyr files
    for root, subdirs, files in os.walk(dir_script):
        for d in subdirs:
            if d == symbology_folder:
                dir_symbology_folder = os.path.join(root, d)

    # Search the symbology layers for this script
    try:
        dir_symbology_folder
        for root, dirs, files in os.walk(dir_symbology_folder):
            for f in files:
                if f == symbolgy_layer_std:
                    lyr_template_std = os.path.join(root, f)
                    print(lyr_template_std)
                if f == symbolgy_layer_perz:
                    lyr_template_perz = os.path.join(root, f)
                    print(lyr_template_perz)

    except NameError:
        print("Symbology folder not found")
        pass

    # Loop through all output folders and apply symbology to each raster
    # The symbology of perz and std is different

    sub_folders = os.listdir(output_rau)
    print(sub_folders)
    for sub in sub_folders:
        if sub.startswith("std"):
            for root, dirs, files in os.walk(os.path.join(output_rau, sub)):
                for f in files:
                    if f.endswith(".tif"):
                        input_file = os.path.join(root, f)
                        tmp_output_file = input_file + "_tmp.lyr"
                        output_file = input_file + ".lyr"
                        lyr_name = f
                        arcpy.MakeRasterLayer_management(input_file, lyr_name)
                        arcpy.SaveToLayerFile_management(lyr_name, tmp_output_file)
                        arcpy.ApplySymbologyFromLayer_management(tmp_output_file, lyr_template_std)
                        arcpy.SaveToLayerFile_management(tmp_output_file, output_file, "RELATIVE")
                        os.remove(tmp_output_file)
        if sub.startswith("perz"):
            for root, dirs, files in os.walk(os.path.join(output_rau, sub)):
                for f in files:
                    if f.endswith(".tif"):
                        input_file = os.path.join(root, f)
                        tmp_output_file = input_file + "_tmp.lyr"
                        output_file = input_file + ".lyr"
                        lyr_name = f
                        arcpy.MakeRasterLayer_management(input_file, lyr_name)
                        arcpy.SaveToLayerFile_management(lyr_name, tmp_output_file)
                        arcpy.ApplySymbologyFromLayer_management(tmp_output_file, lyr_template_perz)
                        arcpy.SaveToLayerFile_management(tmp_output_file, output_file, "RELATIVE")
                        os.remove(tmp_output_file)





# ==============================================================================
# Start processing
# ==============================================================================

if __name__ == "__main__":
    # ==============================================================================
    # Start: Set paths and define parameters
    # ==============================================================================
    # Paths
    # path to dsm point cloud (dsm_laz)
    cloud_dsm = r"D:\F3\forest_structure\DSM_laz"
    # path to output folder
    output_folder = r"D:\F3\forest_structure"

    # coordinate reference system
    # define crs that is used (GK2 = 31466, GK3 = 31467, GK4 = 31468, UTM Zone 32N = 25832, UTM Zone 33N = 25833)
    epsg_code = "25832"

    # Define path to bin folder of LAStools, e.g. C:\LAStools\bin
    lastools = r"C:\LAStools\bin"
    # ==============================================================================
    # End: Set paths and define parameters
    # ==============================================================================

    lascanopy = lastools + r"\\lascanopy.exe"
    lasdiff = lastools + r"\\lasdiff.exe"
    lasgrid = lastools + r"\\lasgrid.exe"

    # Parameters
    cores = multiprocessing.cpu_count() - 1  # define parameters for parallel processing
    pixel_step = [20, 50, 100]  # define geometric resolution (in meters) of output data

    letsgo = worker()
