# ===============================================================================
# file            : modellierung_volumen_biomasse_F3.R
# ===============================================================================

# This R script models timber volume and above-ground biomass based on predictors 
# derived from normalized digital surface models, digital terrain models, soil data,
# climate data and a tree species map. 

# It was devised within the scope of the "F3 - Flchendeckende 
# Fernerkundungsbasierte Forstliche Strukturdaten" (F3- Area-wide remote sensing 
# based forest structural data) project by project partners Forest Research 
# Institute of Baden-Wrttemberg (Forstliche Versuchs- und Forschungsanstalt 
# Baden-Wrttemberg - FVA) and Northwest German Forest Research Institute 
# (Nordwestdeutsche Forstliche Versuchsanstalt - NW-FVA).
# For further information go to www.waldwissen.net/technik/inventur/f3/ or contact
# Petra Adler, Petra.Adler@forst.bwl.de (FVA)
# Jrg Ackermann, Joerg.Ackermann@nw-fva.de (NW-FVA)

# This script is published under GNU General Public License Version 3, 29 June 2007.

# =================================================================================

#################################
# Modellierung von Holzvorrat und Biomasse basierend auf Strukturmetriken (aus Oberfl?chenmodellen), 
# Topographie-Parametern (aus DGM1), Baumartenkarte (aus Sentinel-2-Satellitendaten), Bodendaten (aus soilgrids.org)
# und Klimadaten (DWD). 
# Das Skript liest Rasterdaten verschiedener Quellen ein, die in derselben Projektion und Aufl?sung vorliegen m?ssen. 
# Stichprobenpunkte forstlicher Inventuren mit bekannten Holzvorrats- und Biomassewerten k?nnen als Shape-Dateien eingelesen werden.
# Das Skript f?hrt folgende Schritte durch:
# 1.) Einlesen der Daten
# 2.) Maskieren aller Pixel mit einer Kronen?berschirmung <25%
# 3.) Zuschneiden aller Rasterdaten auf eine gemeinsame Ausdehnung
# 4.) Auslesen der Pixelwerte der Rasterdaten an den Koordinaten der Stichprobenpunkte
# 5.) Detektion und L?schen von Ausrei?ern. Wenn die h?chste Baumh?he eines Stichprobenpunktes um 25%
#     von der h?chsten H?he des Oberfl?chenmodells an diesem Punkt abweicht, gilt dieser Datenpunkt als 
#     Ausrei?er und wird aus dem Datensatz entfernt. 
# 6.) Trainieren je eines Random Forest Modells zur Modellierung von Holzvorrat und Biomasse.
# 7.) Erzeugung von Holzvorrats- und Biomassekarte
# 8.) Berechnung des Vorhersagefehlers (relativer RMSE) des Modells basierend auf OOB Samples. 
#     Graphische Ausgabe eines Scatterplots (vorhegesagte vs. beobachtete Werte)
#####################################

library(raster)
library(Boruta)
library(randomForest)
library(rgdal)

#####################################################

setwd('D:/F3/Daten')

######################################################
# Pfad und Dateiname der Ausgabedateien (Holzvolumen- und Biomassekarte) anpassen.
      map.name.V <- 'Holzvolumen.tif'
      map.name.B <- 'Oberirdische_Biomasse.tif'
      
######################################################
# Einlesen der Stichprobenpunkte (Shape-Datei)
  # Bitte Dateipfad anpassen.
      invPlots <- readOGR('Zusatzdaten/Forst/BI/BI_Hansen/bi_extr_F3_shape.shp')

  # Welche Spalten der Attributtabelle enthalten Holzvolumen, Biomasse und maximale Baumh?he?
    # Bitte Spaltennamen angeben.
      name.holzvol <- 'Vol_ha'
      name.biomasse <- 'oiB_t_ha'
      name.maxheight <- 'max_hoehe'
        
######################################################
# Einlesen der Rasterdaten
  # Bitte Dateipfade zu Strukurparametern anpassen.
      cctot6 <- raster('Abgeleitete_Daten/Aus_Luftbildern/Metriken/cctot6_20_merged_Int16.tif')
      cctot20 <- raster('Abgeleitete_Daten/Aus_Luftbildern/Metriken/cctot20_20_merged_Int16.tif')
      mean20 <- raster('Abgeleitete_Daten/Aus_Luftbildern/Metriken/mean20_merged_Int16.tif')
      perc75 <- raster('Abgeleitete_Daten/Aus_Luftbildern/Metriken/perc75_20_merged_Int16.tif')
      perc95 <- raster('Abgeleitete_Daten/Aus_Luftbildern/Metriken/perc95_20_merged_Int16.tif')
      std20 <- raster('Abgeleitete_Daten/Aus_Luftbildern/Metriken/std_20_merged_Float32.tif')
      max20 <- raster('Abgeleitete_Daten/Aus_Luftbildern/Metriken/max20_merged_Int16.tif')
      
  # Bitte den Dateinamen des Rasters max20 (Raster, das die Hhe des hchsten Punktes des nDSM pro Flcheneinheit enthlt)  
  # angeben (ohne Zusatz '.tif'). 
      max20.filename <- 'max20_merged_Int16'
      
      cctot6[cctot6 < 25] <- NA      
      
      predictor1 <- stack(cctot6, cctot20, mean20, perc75, perc95, std20, max20)
######################################################
  # Bitte Dateipfade zu Topographieparametern anpassen.
      dgm.mean <- raster('Zusatzdaten/DGM/dgm1_mean20m_merged.tif')
      dgm.std <- raster('Zusatzdaten/DGM/dgm1_std20m_merged.tif')
      
      predictor2 <- stack(dgm.mean, dgm.std)
######################################################
  # Bitte Dateipfad zur Baumartenkarte anpassen.   
      tree.spec <- raster('Fernerkundung/Sentinel_2/S2B_MSIL2A_20170823T103019_N0205_R108_T32UND_20170823T103018_Dsen2_TopCorSlope10K06ndvi_VI_HrzSol_BA_Betriebsinv_20m_25832.tif')
  # Bitte Dateinamen der Baumartenkarte angeben (ohne Zusatz '.tif'). 
      tree.spec.filename <- 'S2B_MSIL2A_20170823T103019_N0205_R108_T32UND_20170823T103018_Dsen2_TopCorSlope10K06ndvi_VI_HrzSol_BA_Betriebsinv_20m_25832'
######################################################
  # Bitte Dateipfade zu Bodendaten anpassen. 
      soil.bd <- raster('Zusatzdaten/Standortsdaten/Soilgrids/4_resampled/bulkdensity_max_20m.tif')
      soil.cec1 <- raster('Zusatzdaten/Standortsdaten/Soilgrids/4_resampled/cec1_20m.tif')
      soil.cec2 <- raster('Zusatzdaten/Standortsdaten/Soilgrids/4_resampled/cec2_20m.tif')
      soil.cec3 <- raster('Zusatzdaten/Standortsdaten/Soilgrids/4_resampled/cec3_20m.tif')
      soil.nitogen1 <- raster('Zusatzdaten/Standortsdaten/Soilgrids/4_resampled/nitrogen1_20m.tif')
      soil.nitogen2 <- raster('Zusatzdaten/Standortsdaten/Soilgrids/4_resampled/nitrogen2_20m.tif')
      soil.nitogen3 <- raster('Zusatzdaten/Standortsdaten/Soilgrids/4_resampled/nitrogen3_20m.tif')
      
      predictor3 <- stack(soil.bd, soil.cec1, soil.cec2, soil.cec3, soil.nitogen1, soil.nitogen2, soil.nitogen3)
######################################################
  # Bitte Dateipfade zu Klimadaten anpassen.
      clim.temp <- 'Zusatzdaten/Standortsdaten/Klima/2_resampled_20m_reprojected_25832/RSMS_17_1981_30_temp_max_1981-2010.tif'
      clim.precip <- 'Zusatzdaten/Standortsdaten/Klima/2_resampled_20m_reprojected_25832/TADXMM_17_1981_30_precipitation_1981-2010.tif'
      
      predictor4 <- stack(raster(clim.temp), raster(clim.precip))

######################################################      
######################################################
# crop raster layers to common extent

# create common extent
    ex1 <- extent(predictor1)
    ex2 <- extent(predictor2)
    ex3 <- extent(tree.spec)
    ex4 <- extent(predictor3)
    ex5 <- extent(predictor4)

    ex <- extent(max(ex1[1],ex2[1],ex3[1],ex4[1],ex5[1]),
                 min(ex1[2],ex2[2],ex3[2],ex4[2],ex5[2]),
                 max(ex1[3],ex2[3],ex3[3],ex4[3],ex5[3]),
                 min(ex1[4],ex2[4],ex3[4],ex4[4],ex5[4]))

# crop to common extent
    predictor1 <- crop(predictor1, ex)
    predictor2 <- crop(predictor2, ex)
    tree.spec <- crop(tree.spec, ex)
    predictor3 <- crop(predictor3, ex)
    predictor4 <- crop(predictor4, ex)

###############################
# stack raster layers
    sat <- stack(predictor1, predictor2, tree.spec, predictor3, predictor4)

    plot(sat[[1]])
    plot(invPlots, add=T, pch=20)

###############################
###############################  
# extract values from raster stack
    dat.raw <- extract(sat, invPlots, sp=T, df=F)
    head(dat.raw)
    dat.inv.sat <- dat.raw@data

#################################
# Modellierung
#################################
# read data  
    dat <-  dat.inv.sat
    
    colnames(dat)[colnames(dat) == name.holzvol] <- 'Volumen'
    colnames(dat)[colnames(dat) == name.biomasse] <- 'Biomasse'
    colnames(dat)[colnames(dat) == name.maxheight] <- 'max_height'
    
    datx <- dat[,c('Volumen', 'Biomasse', 'max_height')]
    dat <- dat[,c((dim(dat)[2]+1-dim(sat)[3]) : (dim(dat)[2]))]
    dat <- cbind(datx, dat)
    dat <- na.omit(dat) 

# change tree species to factor
    dat[,which(names(dat)==tree.spec.filename)] <- as.factor(dat[,which(names(dat)==tree.spec.filename)])


#####################
# remove outliers
# all plots in which the max. height of the ndsm deviates more than 25% from highest tree on that plot is identified 
# as an outlier and removed from the dataset.
      dat$h_maxdev <- abs((dat$max_height - dat[,max20.filename]) / dat$max_height)
      dat <- dat[-which(dat$h_maxdev>0.25),]
#######################
#######################
# Modellierung des Holzvorrats
  
      rf.vol <- randomForest(dat$Volumen~., data=dat[,c(4:(dim(dat)[2]-1))], proximity=F, importance=T, keep.forest=T, keep.inbag=T)
      rf.vol
      varImpPlot(rf.vol, type=1)
  
      # prediction error / OOB RMSE
      sqrt(tail(rf.vol$mse, 1)) #  RMSE
      sqrt(tail(rf.vol$mse, 1))/mean(dat$Volumen) #  RMSE%
  
      all <- c(rf.vol$predicted, dat$Volumen)
      minmax <- c(min(all), max(all))
      png(gsub('.tif', '_scatterplot.png', map.name.V), res=100)
      par(pty="s")
      plot(rf.vol$predicted, dat$Volumen, xlim=minmax, ylim=minmax)
      abline(0, 1, col="grey", lwd=3)   
      legend('bottomright', c('RMSE (%):', paste(round(sqrt(tail(rf.vol$mse, 1))/mean(dat$Volumen)*100, digits=2))), bty = 'n')
      legend('topleft', c('RMSE:', paste(round(sqrt(tail(rf.vol$mse, 1)), digits=2))), bty = 'n')
      dev.off()
      
#######################################
# Erstellen der Holzvorratskarte

      map.volume <- predict(sat, rf.vol, progress='text', type='response')
      #colfunc.vol <- colorRampPalette(c("#9C541F", "#FFFFBF", "#218291"))
      #plot(map.volume, col=(colfunc.vol(100)))
      writeRaster(map.volume, map.name.V, overwrite=T)

########################################
# Modellierung der oberirdischen Biomasse

      rf.bio <- randomForest(dat$Biomasse~., data=dat[,c(4:(dim(dat)[2]-1))], proximity=F, importance=T, keep.forest=T, keep.inbag=T)
      rf.bio
      varImpPlot(rf.bio, type=1)

# prediction error / OOB RMSE
      sqrt(tail(rf.bio$mse, 1))#RMSE
      sqrt(tail(rf.bio$mse, 1))/mean(dat$Biomasse)#RMSE%
      
      all <- c(rf.bio$predicted, dat$Biomasse)
      minmax <- c(min(all), max(all))
      png(gsub('.tif', '_scatterplot.png', map.name.B), res=100)
      par(pty="s")
      plot(rf.bio$predicted, dat$Biomasse, xlim=minmax, ylim=minmax)
      abline(0, 1, col="grey", lwd=3)   
      legend('bottomright', c('RMSE (%):', paste(round(sqrt(tail(rf.bio$mse, 1))/mean(dat$Biomasse)*100, digits=2))), bty = 'n')
      legend('topleft', c('RMSE:', paste(round(sqrt(tail(rf.bio$mse, 1)), digits=2))), bty = 'n')
      dev.off()
      
#######################################
# Erstellen der Biomassekarte.
      map.biomass <- predict(sat, rf.bio, progress='text', type='response')
      #colfunc.biom <- colorRampPalette(c("#4D2096", "#FFFFBF", "#147A0B"))
      #plot(map.biomass,col=(colfunc.biom(100)))
      writeRaster(map.biomass, map.name.B, overwrite=T)

########################################    

