# -*- coding: utf-8 -*-
"""
===============================================================================
file            : luecken_aufloesen.py
===============================================================================
This Python-script looks for regions classified as gaps in the gap raster output 
of waldtyp_F3.py and are adjacent to regions classified as open forest. All 
gaps that have a common border with open forest longer than 25% of the gaps 
circumference, are merged into the adjacent region of open forest. In order 
to avoid edge effects, the gap classification before clipping should be used. 

It was devised within the scope of the "F3 - Flächendeckende 
Fernerkundungsbasierte Forstliche Strukturdaten" (F3- Area-wide remote sensing 
based forest structural data) project by project partners Forest Research 
Institute of Baden-Württemberg (Forstliche Versuchs- und Forschungsanstalt 
Baden-Württemberg - FVA) and Northwest German Forest Research Institute 
(Nordwestdeutsche Forstliche Versuchsanstalt - NW-FVA).
For further information go to www.waldwissen.net/technik/inventur/f3/ or contact
Petra Adler, Petra.Adler@forst.bwl.de (FVA)
Jörg Ackermann, Joerg.Ackermann@nw-fva.de (NW-FVA)

This script is published under GNU General Public License Version 3, 29 June 2007.

"""

# ===========================================================================
# Import modules 
# ===========================================================================

import os
import arcpy
from arcpy.sa import *
import shutil

#==============================================================================
# Environment settings
#==============================================================================

arcpy.env.overwriteOutput = True

if arcpy.CheckExtension("Spatial") == "Available":
    arcpy.CheckOutExtension("Spatial")
else:
    arcpy.AddError("Spatial Analyst extension is unavailable. Please check to see if you have the Spatial Analyst License.")


# ==============================================================================
# Define functions
# ==============================================================================

def select_by_attribute(to_select, output, expression):
    """ Select Layer by Attribute
    Select a subset of a layer based on a specific expression.
    """
    select_temp = gdb_path + "\\select_temp"
    arcpy.MakeFeatureLayer_management(to_select, select_temp)
    arcpy.SelectLayerByAttribute_management(select_temp,
                                        "NEW_SELECTION", expression)
    arcpy.CopyFeatures_management(select_temp, output)
    arcpy.Delete_management(select_temp)
 
def select_by_location(in_select, select_feature, overlap_type,
                       selection_type, output):
    """ select layer by location """
    in_select_tmp = gdb_path + "\\in_select_tmp"
    select_feature_tmp = gdb_path + "\\select_feature_tmp"   
     
    arcpy.MakeFeatureLayer_management(in_select, in_select_tmp)
    arcpy.MakeFeatureLayer_management(select_feature, select_feature_tmp)
    arcpy.SelectLayerByLocation_management(in_select_tmp, overlap_type,
                                           select_feature_tmp,"",selection_type)
    arcpy.CopyFeatures_management(in_select_tmp, output)
     
    arcpy.Delete_management(in_select_tmp)
    arcpy.Delete_management(select_feature_tmp) 
     
def join_border_based(gdb_path, orig_raster, out_path, epsg_code):
    """ Join border based
    Join gaps to open forest based on their common border.
    """    
    if epsg_code == "31467":
        prefix = "luecken_bereinigt"
    elif epsg_code == "25832" or epsg_code == "25833" :
        prefix = "luecken_bereinigt_"
        
    poly_all = gdb_path + "\\poly_all"
    gaps = gdb_path + "\\gaps"
    open_f = gdb_path + "\\open_f"
    borders = gdb_path + "\\borders"
    borders2 = gdb_path + "\\borders2"
    join_gaps = gdb_path + "\\join_gaps"
    join_gaps_ras = gdb_path + "\\join_gaps_ras"   
    # Get extent of original raster and set extent in environment settings
    tmp_raster = arcpy.sa.Raster(orig_raster)
    arcpy.env.extent = tmp_raster.extent
    # derive polygons from raster
    arcpy.RasterToPolygon_conversion(in_raster = orig_raster,
                                out_polygon_features = poly_all,
                                simplify = "NO_SIMPLIFY", raster_field = "VALUE")
    select_by_attribute(poly_all, gaps, "\"gridcode\" = 3")
    select_by_attribute(poly_all, open_f, "\"gridcode\" = 1")   
    # add a field for shape length (circumference) to the gaps
    arcpy.AddField_management(gaps, "full_length", "FLOAT")
    arcpy.CalculateField_management(gaps, "full_length",
                                    ("!Shape_Length!"), "PYTHON_9.3", "")    
    # get line features for the common border of open forest and gaps
    arcpy.Intersect_analysis([open_f, gaps], borders, "",
                             "", "LINE")
    # add a field for the percentage of common border with open forest polygon
    arcpy.AddField_management(borders, "cborder", "FLOAT")
    arcpy.CalculateField_management(borders, "cborder",
                                    ("!Shape_Length! /!full_length!"),
                                    "PYTHON_9.3", "")
    select_by_attribute(borders, borders2, ("cborder >= 0.25")) # select all gaps with a common border of a specific length (%)   
    select_by_location(gaps, borders2,'contains', 'NEW_SELECTION', join_gaps)
    # Check whether there have been any gaps selected. If not, no join is necessary and the original raster is copied to output folder
    count_feat_tmp = arcpy.GetCount_management(join_gaps)
    count_feat = int(count_feat_tmp.getOutput(0))
    if count_feat == 0:
        shutil.copy(src = orig_raster, dst = out_path + "\\{}{}".format(prefix, orig_raster[-13:]))
        shutil.copy(src = orig_raster, dst = out_path + "\\{}{}{}".format(prefix, orig_raster[-13:-3],"tfw"))
        shutil.copy(src = orig_raster, dst = out_path + "\\{}{}{}".format(prefix, orig_raster[-13:],".aux.xml"))
        shutil.copy(src = orig_raster, dst = out_path + "\\{}{}{}".format(prefix, orig_raster[-13:],".vat.cpg"))
        shutil.copy(src = orig_raster, dst = out_path + "\\{}{}{}".format(prefix, orig_raster[-13:],".vat.dbf"))
    else:
        # Assign open forest class (value = 1)    
        arcpy.CalculateField_management(join_gaps, "gridcode", "1", "PYTHON_9.3", "")
        # Convert gap polygons that should be joined with open forest to raster
        arcpy.PolygonToRaster_conversion(join_gaps, "gridcode", join_gaps_ras, "", "", "1")
        # Substitute pixel value where join_gaps_ras is not NA (substitute class 3 (gap) with class 1 (open forest)
        output_raster = Con(IsNull(join_gaps_ras),orig_raster,join_gaps_ras)
        output_raster.save(out_path + "\\{}{}".format(prefix, orig_raster[-13:]))

# ===========================================================================
# Set variables and start processing 
# ===========================================================================

# set input directory with gap classification (output of forest_type.py)
in_dir = r'D:\F3\forest_structure\waldtyp'
# define output directory
out_dir = r'D:\F3\forest_structure\gaps_dissolved'
# set spatial reference coordinate system of data
epsg_code = "25832" # define coordinate reference system that is used (GK3 = 31467, UTM Zone 32N = 25832, UTM Zone 33N = "25833")

# get files to be processed
for root, dirs, files in os.walk(os.path.abspath(in_dir)):
    if len(dirs) > 0:
        del(dirs[0:len(dirs)])
    merge_dir_list = [os.path.join(root, file) for file in files if file.endswith(".tif")]

# create output directory and temporary ArcGIS file geodatabase
if not os.path.exists(out_dir):
    os.mkdir(out_dir)
gdb_path = out_dir + "\\tmp.gdb"  # create temporary file geodatabase
arcpy.CreateFileGDB_management(out_dir, "tmp.gdb")
    
for file in merge_dir_list:
    join_border_based(gdb_path, file, out_dir, epsg_code)    
arcpy.Delete_management(gdb_path)        
