# coding=utf-8
"""
===============================================================================
file            : lockere_althoelzer_F3.py
===============================================================================

This Python script finds matured stands with wide spacing between single trees
based on the standard deviation of the normalised digital surface model (nDSM).

It was devised within the scope of the "F3 - Flächendeckende 
Fernerkundungsbasierte Forstliche Strukturdaten" (F3- Area-wide remote sensing 
based forest structural data) project by project partners Forest Research 
Institute of Baden-Württemberg (Forstliche Versuchs- und Forschungsanstalt 
Baden-Württemberg - FVA) and Northwest German Forest Research Institute 
(Nordwestdeutsche Forstliche Versuchsanstalt - NW-FVA).
For further information go to www.waldwissen.net/technik/inventur/f3/ or contact
Petra Adler, Petra.Adler@forst.bwl.de (FVA)
Jörg Ackermann, Joerg.Ackermann@nw-fva.de (NW-FVA)

This script is published under GNU General Public License Version 3, 29 June 2007.

"""

# ===========================================================================
# Import modules 
# ===========================================================================

import os
import arcview
import arcpy
from arcpy.sa import *
from shutil import rmtree
import multiprocessing
from timeit import default_timer as timer
import numpy as np
import cv2
from glob import glob

start = timer()

# ==============================================================================
# Define functions
# ==============================================================================

print("Define functions")

def create_folder(output_f, folder_name):
    if not os.path.isdir(os.path.join(output_f, folder_name)):
        os.mkdir(os.path.join(output_f, folder_name))


def save_function(file, main_folder, sub_folder, name):
    output = file.save(os.path.join(main_folder, sub_folder, "{}".format(name[-13:])))
    return output

# ==============================================================================
# Create and set parameters
# ==============================================================================

print("Set folders")
# Path to input directory: nDSM im .LAZ Format
input_folder =r"D:\F3\forest_structure\nDSM_laz"
# Path to output directory
output_folder = r"D:\F3\forest_structure\lockeres_altholz"
# Path to bin-folder of LAStools, e.g. C:\LAStools\bin
lastools = r"S:\software\LAStools19\bin"
lasgrid = lastools + r"\lasgrid.exe"

folder_name_list = ["0_std", "1_con", "2_focal", "3_reclass", "4_opening", "5_regiongroup", "6_extract",
                    "7_reclass", "8_nibble", "altholz4copy"]

color_map_folder = "color_maps"
color_map_file = "color_lockere_althoelzer.clr"

print("Set parameters")
step = 20
value_std = 7
radius = 2
reclass_value = 0.5
extract_size = 25
prefix = "lockere_althoelzer"

# ==============================================================================
# Environment settings
# ==============================================================================

print("Set environments")
arcpy.env.overwriteOutput = True
arcpy.CheckOutExtension("spatial")
cores = multiprocessing.cpu_count() - 1

# ==============================================================================
# Start processing
# ==============================================================================

print("Create folders")
create_folder(output_folder, "temp_lockere_althoelzer")
temp_folder = os.path.join(output_folder, "temp_lockere_althoelzer")
[create_folder(temp_folder, name) for name in folder_name_list]
create_folder(output_folder, "lockere_althoelzer")

# Calculate std from laz/las files
print("Calculate std from laz or las files")
las_files = glob(os.path.join(input_folder, "*.las"))
laz_files = glob(os.path.join(input_folder, "*.laz"))

if not las_files:
    file_type = laz_files[0][-3:]
else:
    file_type = las_files[0][-3:]

parameters = r" -step {} -otif -stddev -cores {} -odir {}".format(step, cores,
                                                                  os.path.join(temp_folder, folder_name_list[0]))
lastools_query = lasgrid + " -i " + os.path.join(input_folder, "*{}".format(file_type)) + parameters
os.system(lastools_query)

# std > 5 gets value 1, else 0
print("Conditional: set 1 if std > value, else set 0")
dir_list = glob(os.path.join(temp_folder, folder_name_list[0], "*.tif"))

for raster in dir_list:
    out = Con(raster, 1, 0, "VALUE > {}".format(value_std))
    save_function(out, temp_folder, folder_name_list[1], raster)

# mean value calculated for each cell with neighbourhood values using focal statistics
print("Calculate mean value for each cell with the values within the circle")
dir_list = glob(os.path.join(temp_folder, folder_name_list[1], "*.tif"))

for raster in dir_list:
    out = FocalStatistics(raster, "Circle {} CELL".format(radius), "MEAN", "DATA")
    save_function(out, temp_folder, folder_name_list[2], raster)

# reclassify in 0 and 1 depending on the mean value calculated in the last step
print("Reclassify cell (mean) values in 0 and 1 using threshold value")
dir_list = glob(os.path.join(temp_folder, folder_name_list[2], "*.tif"))

for raster in dir_list:
    out = Reclassify(raster, "VALUE", "0 {} 0; {} 1 1".format(reclass_value, reclass_value), "DATA")
    save_function(out, temp_folder, folder_name_list[3], raster)

# Get epsg code
spatial_reference = arcpy.Describe(dir_list[0]).spatialReference
arcpy.env.outputCoordinateSystem = spatial_reference

epsg_code = spatial_reference.factoryCode
if epsg_code is 0:  # custom projections don't have factory code
    spatial_info = spatial_reference.exportToString()

    if 'Zone_32N' in spatial_info:
        epsg_code = 25832
    if 'Zone_33N' in spatial_info:
        epsg_code = 25833
    if 'Zone_2' in spatial_info:
        epsg_code = 31466
    if 'Zone_4' in spatial_info:
        epsg_code = 31468

# opening
print("Opening")
dir_list = glob(os.path.join(temp_folder, folder_name_list[3], "*.tif"))

kernel = cv2.getStructuringElement(cv2.MORPH_RECT, (3, 3))

for im_path in dir_list:
    img = arcpy.Raster(im_path)
    xmin = img.extent.XMin
    ymin = img.extent.YMin
    img_array = arcpy.RasterToNumPyArray(img, lower_left_corner=arcpy.Point(xmin, ymin))
    img_array = np.uint8(img_array)
    opening_cv2 = cv2.morphologyEx(img_array, cv2.MORPH_OPEN, kernel)
    out = arcpy.NumPyArrayToRaster(opening_cv2, lower_left_corner=arcpy.Point(xmin, ymin), x_cell_size=step,
                                   y_cell_size=step)
    save_function(out, temp_folder, folder_name_list[4], im_path)

# group into regions of values, link remembers classification of the cells
print("Group into regions and create link that remembers cell value")
dir_list = glob(os.path.join(temp_folder, folder_name_list[4], "*.tif"))

for raster in dir_list:
    out = RegionGroup(raster, "EIGHT", "WITHIN", "ADD_LINK", "")
    save_function(out, temp_folder, folder_name_list[5], raster)

# extract small groups, they get NA value
print("Extract regions bigger than value, smaller regions get NoData")
dir_list = glob(os.path.join(temp_folder, folder_name_list[5], "*.tif"))

for raster in dir_list:
    out = ExtractByAttributes(raster, "Count >= {}".format(extract_size))
    save_function(out, temp_folder, folder_name_list[6], raster)

# reclassify as 1 and 0 using the value stored in the link
print("Reclassify to 0 and 1 using cell value stored in the link")
dir_list = glob(os.path.join(temp_folder, folder_name_list[6], "*.tif"))

for raster in dir_list:
    out = Reclassify(raster, "LINK", "0 0; 1 1")
    save_function(out, temp_folder, folder_name_list[7], raster)

# assign to the na cells the non-na value in the cell closest to the na cell
print("Assign to the cells with NoData the cell value of their closest cell")
dir_list = glob(os.path.join(temp_folder, folder_name_list[7], "*.tif"))
dir_list2 = glob(os.path.join(temp_folder, folder_name_list[4], "*.tif"))

for num, raster in enumerate(dir_list):
    out = Nibble(dir_list2[num], raster, "DATA_ONLY")
    save_function(out, temp_folder, folder_name_list[8], raster)

# Search the color map for this script
print("Find color map")
# Path to the directory where the script is saved
dir_script = os.path.dirname(os.path.realpath(__file__))

# In this directory, search the directory with the color maps
for root, subdirs, files in os.walk(dir_script):
    for d in subdirs:
        if d == color_map_folder:
            dir_color_maps = os.path.join(root, d)
            print("dir_color_maps: " + str(dir_color_maps))

try:
    dir_color_maps
    for root, dirs, files in os.walk(dir_color_maps):
        for f in files:
            if f == color_map_file:
                color_map = os.path.join(root, f)
except NameError:
    print("Color map folder not found")
    pass

# Add color ramp
dirList = glob(os.path.join(temp_folder, folder_name_list[8], "*.tif"))

try:
    color_map
    for file in dirList:
        arcpy.AddColormap_management(in_raster=file, in_template_raster="",
                                     input_CLR_file=color_map)
    print("Color map added to the rasters")
except NameError:
    print("Color map file not found")
    pass

# Remove buffer
print("epsg is:" + str(epsg_code))

for file in dirList:
    # Get 1*1km extent
    if epsg_code == 31467 or epsg_code == 31466 or epsg_code == 31468:
        xmin = file[-12:-8] + "000"
        ymin = file[-8:-4] + "000"
        file_name = file[-12:-4]
    elif epsg_code == 25832 or epsg_code == 25833:
        xmin = file[-11:-8] + "000"
        ymin = file[-8:-4] + "000"
        file_name = file[-13:-4]
    xmax = int(xmin) + 1000
    ymax = int(ymin) + 1000
    coord = [str(xmin), str(ymin), str(xmax), str(ymax)]

    # Clip forest type and canopy cover
    arcpy.Clip_management(in_raster=file, rectangle=(" ".join(coord)),
                          out_raster=os.path.join(temp_folder, folder_name_list[9], "{}_{}.tif".format(prefix, file_name)))
    # use CopyRaster for ensuring each raster uses the same no data value and pixel type (pixel depth)
    in_ras = os.path.join(temp_folder, folder_name_list[9], "{}_{}.tif".format(prefix, file_name))
    out_ras = os.path.join(output_folder, "lockere_althoelzer", "{}_{}.tif".format(prefix, file_name))
    arcpy.CopyRaster_management(in_ras,out_ras,"","","2","NONE","NONE","2_BIT","NONE","NONE","TIFF","NONE")

# Remove intermediate folders
for folder in folder_name_list:
    for filename in os.listdir(os.path.join(temp_folder, folder)):
        arcpy.Delete_management(os.path.join(temp_folder, folder, filename))

try:
    rmtree(temp_folder)
except WindowsError as e:
    print(e)

end = timer()
print("time: " + str(round((end - start)/60)) + " min")
