# -*- coding: utf-8 -*-
"""
===============================================================================
file            : cloud2ndom_F3.py
===============================================================================
This Python script produces digital surface models (DSM) and normalised digital 
surface models (nDSM) from image matching-based point clouds.

It was devised within the scope of the "F3 - Flächendeckende 
Fernerkundungsbasierte Forstliche Strukturdaten" (F3- Area-wide remote sensing 
based forest structural data) project by project partners Forest Research 
Institute of Baden-Württemberg (Forstliche Versuchs- und Forschungsanstalt 
Baden-Württemberg – FVA) and Northwest German Forest Research Institute 
(Nordwestdeutsche Forstliche Versuchsanstalt – NW-FVA).
For further information go to www.waldwissen.net/technik/inventur/f3/ or contact
Petra Adler, Petra.Adler@forst.bwl.de (FVA)
Jörg Ackermann, Joerg.Ackermann@nw-fva.de (NW-FVA)

This script is published under GNU General Public License Version 3, 29 June 2007.

"""

#==============================================================================
# Import moduls
#==============================================================================

import os
import sys
import subprocess
import multiprocessing
import arcpy
from arcpy.sa import *
from shutil import rmtree, copy
from glob import glob
from timeit import default_timer as timer
import rename_F3 as Rename
import re

#==============================================================================
# Environment settings
#==============================================================================

arcpy.CheckOutExtension("spatial")
arcpy.env.overwriteOutput = True

# ===========================================================
# Define Functions
# ===========================================================

def create_lof(input_dir, name_textfile):
    """ Create and return a list of files as .txt from the files in the input directory"""
    dirList = []
    for root, dirs, files in os.walk(os.path.abspath(input_dir)):
        for pointcloud in files:
            dirList.append(os.path.join(root, pointcloud))
    with open(intermediate + '\\{}.txt'.format(name_textfile), 'w') as f:
        [f.write("{}\n".format(item)) for item in dirList]
    f.close()
    return intermediate + '\\' + name_textfile + ".txt"

def process_query(query):
    """ Invoke a subprocess, run the command and save the result of the subprocess call as an object. If the word error
        is found in this object, the script terminates showing the error"""
    shell_output = subprocess.Popen(query, stderr=subprocess.PIPE, stdout=subprocess.PIPE).communicate()[1]
    if "ERROR" in shell_output:
        sys.exit("Process finished because of " + shell_output.rstrip() + " in command: \n " + query)


start = timer()

# ===========================================================
# Set parameters (user input)
# ===========================================================

# Set input path to point cloud
cloud = r"D:\F3\point_cloud"
# input path to DTM data (.laz format)
dtm = r"D:\F3\DTM_1m"
# set output path for nDSM, DSM (.tif and .las)
output_path = r"D:\F3"
# set path to the lastools directory
lastools = r"S:\software\LAStools19\bin"
# set EPSG code of input data coordinate system
# Gauss-Krueger Zone 2 = 31466, Gauss-Krueger Zone 3 = 31467, Gauss-Krueger Zone 4 = 31468
# ERTS89/UTM Zone 32 N = 25832, ERTS89/UTM Zone 33 N = 25833
epsg = 31467
# set minimum area (in %) required for a tile to be processed
min_area = 10

# ===========================================================
# Predefined input
# ===========================================================

# output path for DSM
dsm = os.path.join(output_path, "DSM")
dsm_laz = os.path.join(output_path, "DSM_laz")

# output path for nDSM
ndsm = os.path.join(output_path, "nDSM")
ndsm_laz = os.path.join(output_path, "nDSM_laz")

# path symbology
symbology_folder = "color_maps"
symbolgy_layer = "template_ndom.tif.lyr"
# Path to the directory where the script is saved
dir_script = os.path.dirname(os.path.realpath(__file__))

# Variables
# lastools locations
lastile = os.path.join(lastools, "lastile.exe")
lasheight = os.path.join(lastools, "lasheight.exe")
lasnoise = os.path.join(lastools, "lasnoise.exe")
las2dem = os.path.join(lastools, "las2dem.exe")
lasgrid = os.path.join(lastools, "lasgrid.exe")
lasthin = os.path.join(lastools, "lasthin.exe")
las2las = os.path.join(lastools, "las2las.exe")
lasinfo = os.path.join(lastools, "lasinfo.exe")
lasduplicate = os.path.join(lastools, "lasduplicate.exe")
lasindex = os.path.join(lastools, "lasindex.exe")

cores = multiprocessing.cpu_count() - 1
buffer = "100"

# get name index depending on coordinate reference system
if epsg == 25832:
    index = "32"
    posdiff = 0
elif epsg == 25833:
    index = "33"
    posdiff = 0
elif epsg in (31466, 31467, 31468):
    index = ""
    posdiff = 1  # there is one digit more in GK than with UTM (when indices 32 and 33 are not considered

# ===========================================================
# Start processing
# ===========================================================

# *** Preparations for Calculations ***
# _________________________________________________________________

# Check out the ArcGIS Spatial extension license and set arcpy environments
arcpy.CheckOutExtension("spatial")
arcpy.env.cellSize = "1"
arcpy.env.overwriteOutput = True

# Create output paths
intermediate = os.path.join(output_path, "intermediate")
os.mkdir(intermediate)
os.mkdir(dsm), os.mkdir(ndsm), os.mkdir(dsm_laz), os.mkdir(ndsm_laz)

# --- Remove empty files ---
for f in os.listdir(cloud):
    if os.path.getsize(os.path.join(cloud, f)) < 1500:
        os.remove((os.path.join(cloud, f)))
        print("Empty dsm files removed")

# --- Rename point clouds ---
Rename.rename_F3(cloud, epsg)
Rename.rename_F3(dtm, epsg)

# --- List common tiles between DTM and point cloud ---
# create list of point clouds/DTM without filename extension and without DSM_/DTM_
point_list = [f[4:-4] for f in os.listdir(cloud) if (f.endswith('.laz')) or (f.endswith('.las'))]
pc_format = os.listdir(cloud)[0][-4:]
print("pc format is: " + str(pc_format))
print("points_list:" + str(point_list))

dtm_list = [f[4:-4] for f in os.listdir(dtm) if (f.endswith('.laz')) or (f.endswith('.las'))]

filelist = glob(os.path.join(dtm, "*.lax"))
for f in filelist:
    os.remove(f)

dtm_format = os.listdir(dtm)[0][-4:]
print("dtm_list:" + str(dtm_list))

# Find tiles that occur in cloud and in dtm
common_list = set(point_list) & set(dtm_list)
common_list = list(common_list)
print("common list:" + str(common_list))
print("length common list: " + str(len(common_list)))

# List tiles in photogrammetric point cloud and dtm with DSM_/DTM_ and without filename extension
cloud_common = ["DSM_" + f for f in common_list]
print("cloud common: " + str(cloud_common))
dtm_common = ["DTM_" + f for f in common_list]
print("dtm common: " + str(dtm_common))

# write list of files in the common list
with open(os.path.join(intermediate, "pc_common_tile_list.txt"), "w") as a:
    for f in common_list:
        filename = os.path.join(cloud, "DSM_" + f + pc_format)
        a.write(str(filename) + '\n')
a.close()

common_tiles_dir = os.path.join(intermediate, "pc_common_tile_list.txt")

# *** Find files containing data for at least 10% of tile area ***
# _________________________________________________________________

# Reduce list of files to those that cover at least 100000 m2 (10 % of the tile area)
print("finding files that are smaller than 10 % of the size of the biggest file to check if their data covers at least"
      "10 % of tile area (100 000 sqm)")
threshold_size = max([os.path.getsize(os.path.join(cloud, f + pc_format)) for f in cloud_common])*0.10

# select laz-files smaller than  10 % of the size of the biggest file to create lasinfo file and check the area covered
# by these files
info_dir = os.path.join(intermediate, "las_info")
os.mkdir(info_dir)
params = ' -nh -nv -cd -nmm '

tiles_lasinfo = []
for f in cloud_common:
    if os.path.getsize(os.path.join(cloud, f + pc_format)) < threshold_size:
        tiles_lasinfo.append(f)

if tiles_lasinfo:
    print("there are some small files: " + str(tiles_lasinfo))
    with open(os.path.join(intermediate, "pc_small_4lasinfo.txt"), "w") as a:
        for f in tiles_lasinfo:
            filename = os.path.join(cloud, f + pc_format)
            a.write(str(filename) + '\n')
    a.close()

    pc_small_4lasinfo_dir = os.path.join(intermediate, "pc_small_4lasinfo.txt")

    lastools_query = lasinfo + ' -lof ' + pc_small_4lasinfo_dir + params + ' -otxt -odir ' + info_dir
    process_query(lastools_query)

    # check the area covered by each laz-file and include in list when area covered >= 100000 m2
    remove_list = []
    covered_area_pat = re.compile('covered area in square units\/kilounits: \d{1,7}\/\d\.\d{2}')
    for tile in os.listdir(info_dir):
        with open(os.path.join(info_dir, tile), "r") as infofile:
            line = next(infofile)
            while line:
                if covered_area_pat.match(line):
                    covered_area = int(line[40:len(line) - 6])
                    if covered_area < 1000000 * min_area / 100:
                        remove_list.append(tile[4:len(tile) - 4])
                    break
                line = next(infofile, None)
        infofile.close()

    # check if there were any "small files" that went through the 10 % area filter
    if not remove_list:
        print("all small tiles cover enough area")
    else:
        print("some small tiles don't cover enough area and will be ignored: " + str(remove_list))
        # Update common list removing tiles that do not cover enough area
        common_list = [tile for tile in common_list if tile not in remove_list]
        print("updated common list after removing files with not enough area cover: " + str(common_list))

        # From the new common list, update cloud and dtm list
        cloud_common = ["DSM_" + f for f in common_list]
        dtm_common = ["DTM_" + f for f in common_list]

    # Update list of files
    with open(os.path.join(intermediate, "pc_tile_list.txt"), "w") as a:
        for f in cloud_common:
            filename = os.path.join(cloud, f + pc_format)
            a.write(str(filename) + '\n')
    a.close()
    common_tiles_dir = os.path.join(intermediate, "pc_tile_list.txt")

else:
    print("no small files")

# *** Ensure equal point spacing for input ***
# _________________________________________________________________

# Check point spacing
print("calculating point spacing")
params = ' -nh -nv -cd -nmm '
lastools_query = lasinfo + ' -i ' + os.path.join(cloud, str(cloud_common[0]) + pc_format) + params + "-o " + \
                 os.path.join(intermediate, 'las_info.txt')
process_query(lastools_query)

point_space = 0.2  # just in case lasinfo does not produce the required information (forcing lasthin)

point_space_pat = re.compile(' {6}spacing: all returns \d\.\d{2} last only \d\.\d{2} \(in units\)\\n')
with open(intermediate + '\las_info.txt', "r") as infofile:
    line = next(infofile)
    while line:
        if point_space_pat.match(line):
            point_space = float(line[27:31])
            break
        line = next(infofile, None)
infofile.close()

print("point spacing of {} m extracted using lasinfo".format(point_space))

# Produce point clouds with point spacing = 0.5 if necessary
# From output create a list of files for the next step
if point_space < 0.5:
    print("thinning point clouds")
    lasthin_dir = os.path.join(intermediate, "lasthin")
    os.mkdir(lasthin_dir)
    # execute lasthin, use only point clouds where a DTM exists
    params = ' -percentile 95 -step 0.5 -cores {} -olaz '.format(cores)
    lastools_query = lasthin + ' -cpu64' + ' -lof ' + common_tiles_dir + params + '-odir ' + lasthin_dir
    process_query(lastools_query)

    pc_list4class = create_lof(lasthin_dir, "pc_list4class")
    thinning = True
else:
    pc_list4class = os.path.join(intermediate,
                                 "pc_tile_list.txt")  # no need to create a new lof because nothing happened
    thinning = False
print("lasthin done")

# *** Create buffered tiles ***
# _________________________________________________________________

las2las5_dir = os.path.join(intermediate, "class_5")
os.mkdir(las2las5_dir)

# Assign class 5 to all points in point cloud
params = ' -set_classification 5 -olaz -cores {} '.format(cores)
lastools_query = las2las + ' -lof ' + pc_list4class + params + '-odir ' + las2las5_dir
process_query(lastools_query)
print("las2las class 5 done")

# Prepare files for lastile
lastile_pre = os.path.join(intermediate, "files_for_tiling")
dtm_pre = os.path.join(intermediate, "dtm_for_tiling")
os.mkdir(lastile_pre)
os.mkdir(dtm_pre)

# only process dtm and dsm files that are in the common list
pc_format = os.listdir(las2las5_dir)[0][-4:]
for f in cloud_common:
    tile_origin = os.path.join(las2las5_dir, f + pc_format)
    tile_new_dir = os.path.join(lastile_pre, "tile_" + f + pc_format)
    copy(tile_origin, tile_new_dir)

for f in dtm_common:
    dtm_origin = os.path.join(dtm, f + dtm_format)
    dtm_new_dir = os.path.join(dtm_pre, "tile_" + f + dtm_format)
    copy(dtm_origin, dtm_new_dir)

# lasindex to run lastile on multiple cores

lastools_query = lasindex + ' -i ' + os.path.join(lastile_pre, "*" + pc_format)
process_query(lastools_query)
print("first lasindex done")

lastools_query = lasindex + ' -i ' + os.path.join(dtm_pre, "*" + dtm_format)
process_query(lastools_query)
print("second lasindex done")

# Combine photogrammetry and DTM points and create 1000x1000m tiles of point clouds (include 100 m buffer)
lastile_dir = os.path.join(intermediate, "lastile")
os.mkdir(lastile_dir)

params = ' -tile_size 1000 -buffer {} -rescale 0.01 0.01 0.01 -flag_as_withheld -epsg {} -cores {} '.format(buffer,
                                                                                                            str(epsg),
                                                                                                            cores)
lastools_query = lastile + ' -i ' + os.path.join(lastile_pre, "*" + pc_format) + ' -i ' + \
                 os.path.join(dtm_pre, "*" + dtm_format) + params + '-odir ' + lastile_dir
process_query(lastools_query)
print("lastiles done")

# Rename tiles
for filename_old in os.listdir(lastile_dir):
    filename_new = "tile_" + index + filename_old[0:3 + posdiff] + filename_old[
                                                                   7 + posdiff:11 + posdiff] + filename_old[-4:]
    fpath = os.path.join(lastile_dir, filename_old)
    outpath = os.path.join(lastile_dir, filename_new)
    os.rename(fpath, outpath)
print("rename done")

# *** Normalisation and filtering ***
# _________________________________________________________________

# Height normalisation
lasheight_dir = os.path.join(intermediate, "lasheight")
os.mkdir(lasheight_dir)

params = ' -class 0 -drop_below -1 -drop_above 55 -replace_z -cores {} '.format(cores)
lastools_query = lasheight + ' -lof ' + create_lof(lastile_dir, "list4lasheight") + params + '-odir ' + lasheight_dir
process_query(lastools_query)
print("lasheight done, normalised")

# remove DTM points
las2las_dir = os.path.join(intermediate, "las2las")
os.mkdir(las2las_dir)

params = ' -drop_classification 0 -cores {} '.format(cores)
lastools_query = las2las + ' -lof ' + create_lof(lasheight_dir, "list4las2las") + params + '-odir ' + las2las_dir
process_query(lastools_query)
print("las2las done dtm removed")

# classify all points above 75 percentile into class 8
lasthin2_dir = os.path.join(intermediate, "lasthin2")
os.mkdir(lasthin2_dir)

params = ' -percentile 75 -step 4 -classify_as 8 -cores {} '.format(cores)
lastools_query = lasthin + ' -lof ' + create_lof(las2las_dir, "list4lasthin2") + params + '-odir ' + lasthin2_dir
process_query(lastools_query)
print("lasthin done percentile 75")

lasheight2_dir = os.path.join(intermediate, "lasheight2")
os.mkdir(lasheight2_dir)

params = ' -class "8" -classify_below 0 3 -cores {} '.format(cores)
lastools_query = lasheight + ' -lof ' + create_lof(lasthin2_dir, "list4lasheight2") + params + '-odir ' + lasheight2_dir
process_query(lastools_query)
print("lasheight done class 8")

# further filtering of point cloud using lasnoise (twice)
lasnoise_dir = os.path.join(intermediate, "lasnoise")
os.mkdir(lasnoise_dir)

params = ' -ignore_class 3 -step_xy 10 -step_z 4 -isolated 40 -remove_noise -cores {} '.format(cores)
lastools_query = lasnoise + ' -lof ' + create_lof(lasheight2_dir, "list4lasnoise") + params + '-odir ' + lasnoise_dir
process_query(lastools_query)
print("first lasnoise done")

params = ' -ignore_class 3 -step_xy 3 -step_z 3 -isolated 8 -remove_noise -cores {} -olaz '.format(cores)
lastools_query = lasnoise + ' -lof ' + create_lof(lasnoise_dir, "list4lasnoise2") + params + '-odir ' + ndsm_laz
process_query(lastools_query)
print("second lasnoise done")

# Change name output ndsm laz
for filename_old in os.listdir(ndsm_laz):
    filename_new = "ndsm_" + filename_old[5:]
    fpath = os.path.join(ndsm_laz, filename_old)
    outpath = os.path.join(ndsm_laz, filename_new)
    os.rename(fpath, outpath)

# *** Calculate DSM ***
# _________________________________________________________________

# Calculation of DSM in .laz format
# Put filtered ndsm and dsm together
lastile2_dir = os.path.join(intermediate, "lastile2")
os.mkdir(lastile2_dir)

lastools_query = lasindex + ' -i ' + os.path.join(ndsm_laz, "*" + dtm_format)
os.system(lastools_query)

lastools_query = lasindex + ' -i ' + os.path.join(lastile_pre, "*" + dtm_format)
os.system(lastools_query)

params = ' -cores {} -drop_withheld '.format(cores)
lastools_query = lastile + ' -i ' + os.path.join(ndsm_laz, "*" + dtm_format) + ' -i ' + \
                 os.path.join(lastile_pre, "*" + pc_format) + \
                 params + '-odir ' + lastile2_dir
process_query(lastools_query)

for filename_old in os.listdir(lastile2_dir):
    filename_new = "tile_" + index + filename_old[0:3 + posdiff] + filename_old[
                                                                   7 + posdiff:11 + posdiff] + filename_old[-4:]
    fpath = os.path.join(lastile2_dir, filename_old)
    outpath = os.path.join(lastile2_dir, filename_new)
    os.rename(fpath, outpath)

# Only points with same x and y as in ndsm remain in dsm
params = ' -record_removed -odix _rm -drop_withheld -cores {} -olaz '.format(cores)
lastools_query = lasduplicate + ' -lof ' + create_lof(lastile2_dir, "list4lasduplicate") + params + '-odir ' + dsm_laz
process_query(lastools_query)
print("lasduplicate done")

# remove dsm output that is not needed
filelist = glob(os.path.join(ndsm_laz, "*.lax"))
for f in filelist:
    os.remove(f)

for fname in os.listdir(dsm_laz):
    if fname[-7:-4] == "_rm":
        os.remove(os.path.join(dsm_laz, fname))

# Rename dsm files
for filename_old in os.listdir(dsm_laz):
    filename_new = "dsm_" + filename_old[5:-15] + filename_old[-4:]
    fpath = os.path.join(dsm_laz, filename_old)
    outpath = os.path.join(dsm_laz, filename_new)
    os.rename(fpath, outpath)

# *** Rasterize nDSM and DSM ***
# _________________________________________________________________

# Calculation of DSM and nDSM in tif-format, small NA areas will be interpolated and filled
lasgrid_dir = os.path.join(intermediate, "lasgrid")
os.mkdir(lasgrid_dir)

params = ' -elevation -highest -drop_withheld -fill 3 -step 1 -cores {} -clamp_z_below 0 -otif '.format(cores)
lastools_query = lasgrid + ' -lof ' + create_lof(ndsm_laz, "list4lasgrid") + params + '-odir ' + lasgrid_dir
process_query(lastools_query)

# Check if GRID has NoData pixels
# execute ArcGis Tool GetRasterProperties
las2dem_dir = os.path.join(intermediate, "las2dem")
os.mkdir(las2dem_dir)

lasgrid_list = []
las2dem_list = []
ndsm_list = []

for tile in common_list:
    lasgrid_list.append(os.path.join(lasgrid_dir, "ndsm_" + tile + ".tif"))
    las2dem_list.append(os.path.join(las2dem_dir, "ndsm_" + tile + ".tif"))
    ndsm_list.append(os.path.join(ndsm_laz, "ndsm_" + tile + dtm_format))

for index, raster in enumerate(iter(lasgrid_list)):
    no_data = arcpy.GetRasterProperties_management(raster, "ANYNODATA").getOutput(0)

    if no_data == u'0':
        print("No need for triangulation")
        copy(raster, ndsm)

    if no_data == u'1':
        print("Calculate TIN")

        params = ' -elevation -step 1 -use_tile_bb -clamp_z_below 0 '
        lastools_query = las2dem + ' -i ' + ndsm_list[index] + params + '-odir ' + las2dem_dir + ' -otif'
        process_query(lastools_query)
        arcpy.DefineProjection_management(raster, epsg)

        # Calculate dsm
        # set paths
        las2dem_file = las2dem_list[index]

        # Raster calculator 1: Combine GRID and NDSM
        OutRaster1 = Con(IsNull(raster), las2dem_file, raster)
        arcpy.DefineProjection_management(raster, epsg)
        OutRaster1.save(os.path.join(ndsm, "ndsm_" + raster[-13 + posdiff:]))

    arcpy.DefineProjection_management(raster, epsg)

# execute lasgrid on dtm
lasgrid2_dir = os.path.join(intermediate, "lasgrid2")
os.mkdir(lasgrid2_dir)

with open(os.path.join(intermediate, "dtm_tile_list.txt"), "w") as a:
    for f in common_list:
        filename = os.path.join(dtm, "DTM_" + f + dtm_format)
        a.write(str(filename) + '\n')
a.close()

dtm_tiles_list = os.path.join(intermediate, "dtm_tile_list.txt")

params = ' -elevation -highest -fill 3 -cores {} -step 1 -otif '.format(cores)
lastools_query = lasgrid + ' -lof ' + dtm_tiles_list + params + '-odir ' + lasgrid2_dir
process_query(lastools_query)

# Calculate DSM raster = nDSM + DTM
dsm_temp_dir = os.path.join(intermediate, "dsm_temp")
os.mkdir(dsm_temp_dir)

ndsm_list = []

for tile in common_list:
    ndsm_list.append(os.path.join(ndsm, "ndsm_" + tile + ".tif"))

for raster in ndsm_list:
    dtm_tile = "DTM_" + raster[-13 + posdiff:]
    dsm_tile = "dsm_" + raster[-13 + posdiff:]

    dtm_file = os.path.join(lasgrid2_dir, dtm_tile)

    OutRaster1 = Raster(raster) + Raster(dtm_file)

    dsm_temp = os.path.join(dsm, dsm_tile)
    OutRaster1.save(dsm_temp)

# *** Apply color map ***
# _________________________________________________________________

# Apply Symbology
input_files = glob(os.path.join(ndsm, "*.tif"))

# In this directory, search the directory with the lyr file
for root, subdirs, files in os.walk(dir_script):
    for d in subdirs:
        if d == symbology_folder:
            dir_symbology_folder = os.path.join(root, d)

# Search the color map for this script
try:
    dir_symbology_folder
    for root, dirs, files in os.walk(dir_symbology_folder):
        for f in files:
            if f == symbolgy_layer:
                lyr_file = os.path.join(root, f)
except NameError:
    print("Color map folder not found")
    pass

for file in input_files:
    print(file)
    temp_file = "tmp_" + file[-13 + posdiff:-4]
    tmp_output_file = file + "_tmp.lyr"
    print(tmp_output_file)
    output_file = file + ".lyr"
    arcpy.MakeRasterLayer_management(file, temp_file)
    arcpy.SaveToLayerFile_management(temp_file, tmp_output_file)
    arcpy.ApplySymbologyFromLayer_management(tmp_output_file, lyr_file)
    arcpy.SaveToLayerFile_management(tmp_output_file, output_file, "RELATIVE")
    os.remove(tmp_output_file)

# *** Remove intermediate data that is no longer needed ***
# _________________________________________________________________

print("Deleting intermediate data...")

folder_name_list = ['class_5', 'dtm_for_tiling', 'files_for_tiling', 'las2dem', 'las2las', 'lasgrid', 'lasgrid2',
                    'lasheight',
                    'lasheight2', 'lasnoise', 'lasthin', 'lasthin2', 'lastile', 'lastile2', 'dsm_temp', 'las_info']

if thinning:
    pass
else:
    folder_name_list.remove('lasthin')

for folder in folder_name_list:
    for filename in os.listdir(os.path.join(intermediate, folder)):
        arcpy.Delete_management(os.path.join(intermediate, folder, filename))

try:
    rmtree(intermediate)
except WindowsError as e:
    print(e)

end = timer()
print("ended in " + str(round((end - start) / 60)) + " minutes")
