# ===============================================================================
# file            : S2_topocorr.R
# ===============================================================================

# This R script processes Sentinel-2 satellite data in order to prepare the images
# for tree species classification. This script will execute topographic corrections,
# mask non-vegetation areas, clouds, cloud shadow and snow als well as calculate
# vegetation indices. (Author: Philip Beckschfer; Email: philip.beckschaefer@gmail.com)

# It was devised within the scope of the "F3 - Flchendeckende 
# Fernerkundungsbasierte Forstliche Strukturdaten" (F3- Area-wide remote sensing 
# based forest structural data) project by project partners Forest Research 
# Institute of Baden-Wrttemberg (Forstliche Versuchs- und Forschungsanstalt 
# Baden-Wrttemberg - FVA) and Northwest German Forest Research Institute 
# (Nordwestdeutsche Forstliche Versuchsanstalt - NW-FVA).
# For further information go to www.waldwissen.net/technik/inventur/f3/ or contact
# Petra Adler, Petra.Adler@forst.bwl.de (FVA)
# Jrg Ackermann, Joerg.Ackermann@nw-fva.de (NW-FVA)

# This script is published under GNU General Public License Version 3, 29 June 2007.

# =================================================================================

pckgs <- c("raster","RStoolbox", "plotKML")
if (length(setdiff(pckgs, rownames(installed.packages()))) > 0) {
  install.packages(setdiff(pckgs, rownames(installed.packages())))  
}

library(raster)
library(RStoolbox)
library(plotKML)

#######################################################
# Bitte setzten Sie die korrekten Pfade und spezifizieren 
# Sie die geforderten Parameter!
#######################################################
        # Pfad zur Working Directory
        setwd('D:/F3/Daten/Sentinel_2')
    
        # Pfad zum digitalen Gelndemodell
        dgm <- raster('D:/F3/Daten/Zusatzdaten/DGM/dgm10.tif')
    
        # Pfad zur Sentinel-2 Szene ohne Dateieindung ".tif".
        # Das topographisch korrigierte Sentinel-2 Bild wird unter dem selben Pfad 
        # mit dem Namenszusatz '_TC.tif' gespeichert.
        s2.path <- 'S2A_sen2cor_noTopCor/S2B_MSIL2A_20170823T103019_N0205_R108_T32UNC_20170823T103018_Dsen2'    
        
        # Pfad zur Wolkenmaske, die mit der Software Fmask 4.0 erstellt wurde.
        cloud <- raster('Fmask/L1C_T32UNC_A002422_20170823T103018_Fmask.tif')
    
        # sun zenith und sun azimuth der Sentinel-2 Szene
        sun.z <- 50.25  # sun zenith
        sun.a <- 160.49 # sun azimuth
    
        # Sollen Vegetationsindices berechnet werden? TRUE or FALSE.
        # Vegetationsindices werden nach der topographischen Korrektur berechnet und
        # unter dem oben angegebnen Pfad (s2.path) mit dem Namenszusatz  
        # '_TC_VI1.tif' bzw. '_TC_VI2.tif' gespeichert.
        calculate_vegetation_indices <- TRUE
        
        # Soll eine NDVI basierte Vegetationsmaske erstellt werden 
        # um nicht-vegetationsflchen aus dem Sentinel-2 Bild zu entfernen? 
        # TRUE or FALSE
        aplly_ndvimask <- TRUE
        
        # Sind die Pixelwerte des Sentinel-2 Bildes zwischen 0 und 1 skaliert? TRUE or FALSE 
        val_0_1 <- FALSE
        
        # Die Prozessierung eines Seninel-2 Bildes mit der Software DSen2 ndert die Reihenfolge
        # der spektralen Bnder. Zur korrekten Erstellung einer NDVI-Maske und zur Berechnung 
        # von Vegegationsindizes mssen die Bnder wieder in ihre ursprngliche Reihenfolge 
        # gebracht werden.
        # Wurden das Sentinel-2 Bild mit der Software DSen2 prozessiert?
        # TRUE or FALSE
        rearrange_s2_bands <- TRUE

# Nachdem alle Pfade und Parameter gesetzt wurden, kann das Skript gestartet werden.
# Im folgenden Code mssen keine nderungen vorgenommen werden!
    
##################################################
# Teil I
# Vorbereitung des digitalen Gelndemodells (DGM)
# Zuschneiden und resamplen des DGM auf Ausdehnung und Auflsung der Sentinel-2 Szene
##################################################
    
      # Lade Sentinel-2 daten
        s2 <- stack(paste(s2.path,'.tif', sep=''))
        
      # Reprojizieren des DGM
        if(compareCRS(dgm ,s2)){
            print("DTM and Sentinel 2 data use the same coordinate reference system.")
          }else{
            dgm <- reproject(dgm, s2@crs)
          }
      # Zuschneiden des DGM   
          if(compareRaster(dgm, s2, extent=TRUE, rowcol=F, crs=F, rotation=F, stopiffalse=F)){
            print("DTM and Sentinel 2 data have the same extent.")
             }else{
               print("DTM is cropped to the extent of the Sentinel-2 data.")
               ex <- extent(s2)
               dgm <- crop(dgm, ex, snap='out')
             }
      # Resampling des DGM      
          if(compareRaster(dgm, s2, extent=F, rowcol=F, crs=F, rotation=F, res=TRUE, stopiffalse=F)){
              print("DTM and Sentinel 2 data have the same resolution.")
            }else{
              print("DTM is resampled to the resolution of the Sentinel-2 data.")
              dgm <- resample(dgm, s2[[1]])
            }
             
      # clear memory
          gc()

#################
# Teil II - Topgrapic correction of the Sentinel-2 image
# All Sentinel-2 Images which have the same extent as the above generated terrain model
# can be processed with the following code. If you wish to process Sentinel-2 images
# from a different location, Part I of the Script needs to be run for this new location first 
# so that the terrain model is prepared correctly. 
#################

      # load S2 image  
      s2 <- stack(paste(s2.path, '.tif', sep=''))
      
      # rearrange s2 image bands  
      if(rearrange_s2_bands == TRUE){
          s2 <- stack(s2[[3]], s2[[2]], s2[[1]], s2[[5]], s2[[6]], s2[[7]], s2[[4]], s2[[8]], s2[[9]], s2[[10]])
          }
      
      # Umrechunung von sun zenith und azimuth zu Radians 
      sun.z <- (sun.z * pi) / 180 
      sun.a <- (sun.a * pi) / 180 
      
      # reclassify and resample cloud mask produced with Fmask 4.0    
      if(exists("cloud")){
          cloud <- reclassify(cloud, c(-Inf, 1, 1,  1, 4, NA), right=TRUE) # mask clouds, shadows and snow
          cloud <- resample(cloud, s2[[1]], method = 'ngb')
          }
      
      # Berrechnung des NDVI
      if(aplly_ndvimask == TRUE){
        ndvi <- ((s2[[7]]-s2[[3]])/(s2[[7]]+s2[[3]]))
        ndvi1 <- reclassify(ndvi, c(-Inf, 0.6, NA,  0.6, Inf, 1), right=TRUE) 
        cloud <- cloud * ndvi
        }
    
      # mask cloud areas from S2 image
      if(exists("cloud")){
          s2 <- s2*cloud
            }else{
              s2 <- s2*ndvi
            }
      
      # if the image has values between 0 and 1 it needs to be multiplied by 10000. This rescalaes the values between 0 and 10000 and allows to save the image as 'INT2S'-> small file size.  
      if(val_0_1 == TRUE){
            s2 <- s2*10000 
            }
     
      rm(ndvi, clndvi, cloud); gc()
      
      # Der folgende Code Block definiert die fr die topographische Korrketur ntige 
      # Funktion. Der verwendete Code stammt aus Funktion topCor() des  R Paket RStoolbox 
      # von B.Leutner: http://bleutner.github.io/RStoolbox/
      # Der Code wurde angepasst um eine berkorrktur der S2 Bilder zu vermeiden.
      ###########################################
      #' Topographic Illumination Correction
      #' 
      #' account and correct for changes in illumination due to terrain elevation.
      #' 
      #' @param img Raster*. Imagery to correct
      #' @param dem Raster*. Either a digital elevation model as a RasterLayer or a RasterStack/Brick with pre-calculated slope and aspect (see \link[raster]{terrain}) in which case the layers must be named 'slope' and 'aspect'. 
      #' Must have the same dimensions as \code{img}.
      #' @param metaData Character, ImageMetaData. Either a path to a Landsat meta-data file (MTL) or an ImageMetaData object (see \link{readMeta}) 
      #' @param solarAngles Numeric vector containing sun azimuth and sun zenith (in radians and in that order). Not needed if metaData is provided   
      #' @param method Character. One of c("cos", "avgcos", "minnaert", "C", "stat", "illu"). Choosing 'illu' will return only the local illumination map.
      #' @param stratImg RasterLayer to define strata, e.g. NDVI. Or the string 'slope' in which case stratification will be on \code{nStrat} slope classes. Only relevant if \code{method = 'minnaert'}.
      #' @param nStrat Integer. Number of bins or quantiles to stratify by. If a bin has less than 50 samples it will be merged with the next bin. Only relevant if \code{method = 'minnaert'}.
      #' @param illu Raster*. Optional pre-calculated ilumination map. Run topCor with method="illu" to calculate an ilumination map
      #' @param ... arguments passed to \code{\link[raster]{writeRaster}}
      #' @details
      #' For detailed discussion of the various approaches please see Riano et al. (2003).
      #' 
      #' The minnaert correction can be stratified for different landcover characteristics. If \code{stratImg = 'slope'} the analysis is stratified by the slope, 
      #' i.e. the slope values are divided into \code{nStrat} classes and the correction coefficient k is calculated and applied separately for each slope class.
      #' An alternative could be to stratify by a vegetation index in which case an additional raster layer has to be provided via the \code{stratImg} argument.
      #' @export 
      #' @references 
      #' Riano et al. (2003) Assessment of different topographic correction in Landsat-TM data for mapping vegetation types. IEEE Transactions on Geoscience and Remote Sensing.
      #' @examples 
      #' ## Load example data
      #' metaData <- system.file("external/landsat/LT52240631988227CUB02_MTL.txt", package="RStoolbox")
      #' metaData <- readMeta(metaData)
      #' lsat     <- stackMeta(metaData) 
      #' data(srtm)
      #' \dontshow{
      #' data(lsat)
      #' }
      #' 
      #' ## Minnaert correction, solar angles from metaData
      #' lsat_minnaert <- topCor(lsat, dem = srtm, metaData = metaData, method = "minnaert")
      #' 
      #' ## C correction, solar angles provided manually
      #' lsat_C <- topCor(lsat, dem = srtm, solarAngles = c(1.081533, 0.7023922), method = "C")
      #' 
      topCor_beckschaefer <- function(img, dem, metaData, solarAngles = c(), method = "minnaert", stratImg = 'slope', nStrat = 10, illu, ...){
        
        stopifnot(method %in% c("cos", "avgcos", "minnaert", "C", "stat", "illu"))
        ## TODO: improve performance
        ## Metadata 
        if(!missing("solarAngles")) {
          if(length(solarAngles)!=2) stop ("If metaData is used to provide solar azimuth and solar zenith it must be a numeric vector of length 2: c(azimuth, zenith)")
          sa <- solarAngles[1]
          sz <- solarAngles[2]
        } else { 
          if(missing("metaData")) stop("You must specify either solarAngles or metaData")
          if(inherits(metaData, "character"))   metaData <- readMeta(metaData)
          sz <- (90-metaData$SOLAR_PARAMETERS[2])*pi/180
          sa <-  metaData$SOLAR_PARAMETERS[1]*pi/180        
        }  
        
        ## Terrain
        if(any(!names(dem) %in% c("slope", "aspect"))) {
          compareRaster(img, dem)
          #.vMessage("Calculate slope and aspect")
          topo <- terrain(dem, c("slope", "aspect"))
        } else {
          compareRaster(img, dem)
          topo <- dem
          ##.vMessage("Using pre-calculated slope and aspect")
        }
        slope <- topo[["slope"]]
        aspect <- topo[["aspect"]]
        
        ## Illumination
        if(missing(illu)){
          #.vMessage("Calculate illumination map")
          illu  <- raster::overlay(topo, fun = function(slope, aspect, sazimuth = sa, szenith = sz){
            cos(szenith) * cos(slope) + sin(szenith) * sin(slope) * cos(sazimuth - aspect)
          })
          names(illu) <- "illu"
          illu <- reclassify(illu, c(-Inf, 0, 0.001), right=TRUE)
        } else {
          #.vMessage("Using pre-calculated illumination map")
        }
        if(method=="illu") return(illu)
        
        #.vMessage("Correct imagery")
        if (method == "cos") {
          ## valid range: <55 degree
          ## Eq 3 in Riano2003
          ## Lambertian assumption              
          Lh <- raster::overlay(img, illu, fun= function(x,y){x * (cos(sz) / y)}, forcefun = TRUE, ...)      
        }
        if (method == "avgcos") {
          ## Eq 4 in Riano2003
          ## Lambertian assumption
          avgillu <- cellStats(illu, mean)
          Lh <- overlay(img, illu, fun= function(x,y){ x + x * (avgillu-y) / avgillu}, forcefun = TRUE, ...)  
        }
        if(method =="minnaert") {
          ## Eq 5 in Riano2003
          ## Lambertian assumption if k == 1
          ## Non-lambertian if 0 <= k < 1   
          stratMethod <- if(is.null(stratImg)) {stratImg <- "slope"; "noStrat"} else "stratEqualBins"
          ks <- .kestimate(img, illu, slope, method = stratMethod, stratImg = stratImg, n = nStrat, sz=sz)
          
          ks$k <- lapply(ks$k, function(x){
            x[x[,2] < 0, 2] <- 0
            x[x[,2] > 1, 2] <- 1
            x
          })
          print(ks)
          Lh <- lapply(1:nlayers(img), function(layer){ overlay(stack(img[[layer]], illu, slope), fun = function(img, illu, strat, groups = ks$groups, k = ks$k) {
            sc <- cut(strat, breaks = groups, labels = FALSE)
            k <- k[[layer]][sc,2] 
            Lh <- img * c(cos(sz)/ illu)^(k*0.6) 
          })
          })
          Lh <- stack(Lh)  
          ellip <- list(...)
          if ('filename' %in% names(ellip) && !is.null(ellip[["filename"]])) {
            names(Lh) <- names(img)
            Lh <- writeRaster(Lh, ...)
          }
        }    
        if(method ==  "stat") {
          ## Eq 8 in Riano2003        
          ks <- .kestimate(img, illu, slope, method = "stat")
          sub <- stack(lapply(ks$k, function(x){
            x[,2] * illu
          }))
          Lh <- overlay(img, sub, fun = function(x,y) {x-y}, ..., forcefun = TRUE)
        }
        if(method == "C") {
          ks <- .kestimate(img, illu, slope, method = "stat")
          mult <- stack(lapply(ks$k, function(x){
            ck <- x[,1]/x[,2] 
            (cos(sz) + ck) /  (illu + ck)
          })) 
          Lh <- overlay(img, mult, fun = function(x,y) {x * y}, forcefun = TRUE, ...)
        }
        
        names(Lh) <- names(img)
        return(Lh)
        #    if(FALSE && method == "minnaMod"){
        #        ## Richter 2009
        #        if(sz < 45*pi/180) {
        #            beta_t <- sz + 20*pi/180
        #        } else if(sz > 55*pi/180) {
        #            beta_t <- sz + 10*pi/180        
        #        } else {
        #            beta_t <- sz + 15*pi/180
        #        }
        #        
        #        ## Vegetation classes: 1 = non-veg, 2 = veg
        #        bvis = c(0.5,  ## non-vegetation
        #                3/4)  ## vegetation (wavelength < 720nm)
        #        bir = c(0.5,  ## non-vegetation
        #                1/3)  ## vegetation (wavelength > 720nm)
        #   
        #        minnaMod <- function(x, beta_tx, b_lut) {
        #            b      <- b_lut[x[,2]]
        #            mult <- (x[,1]/beta_tx)^b
        #            mult[mult > 0.25] <- 0.25
        #            mult
        #        }
        #        
        #     multvis <- calc(stack(illu, stratImg), fun = function(x) minnaMod(x, b_lut = bvis, beta_tx = beta_t))
        #     multir  <- calc(stack(illu, stratImg), fun = function(x) minnaMod(x, b_lut = bir, beta_tx = beta_t))
        #      
        #     select <- illu > beta_t 
        #     visBands <- 1:4    
        #     visCo <- img[visBands]
        #     visCo[select] <- img[[visBands]][select]  * multvis[select]
        #    }
        
      }
      
      
      
      #' Parameter estimation
      #' @noRd 
      #' @keywords internal
      .kestimate <- function(img, illu, slope, stratImg = "slope", method = "noStrat", n = 5, minN = 50, sz) {
        
        stopifnot(method %in% c("stat", "noStrat", "stratEqualBins", "stratQuantiles"))
        ## Following Lu 2008 sample pre selection
        set.seed(10)
        strat <- if(inherits(stratImg, "character")) NULL else {names(stratImg) <- "strat"; stratImg} 
        sr       <- as.data.frame(sampleRandom(stack(img, illu, slope, strat), size = 10000))
        
        if(method != "stat") sr  <- sr[sr$slope > 2*pi/180 & sr$illu >= 0,]
        if(method != "noStrat" & inherits(stratImg, "character")) {
          sr$strat <- sr[,stratImg]
          stratImg <- slope
        }
        
        if(method %in% c("stat","noStrat")){
          groups <- 0:1
          assoc <- rep(1, length(nrow(sr)))
        } else {
          #.vMessage("Begin strafification")
          if(method == "stratQuantiles") {
            ## Quantile method
            groups <- quantile(stratImg, probs = 0:n/n)
          } 
          if(method == "stratEqualBins") {
            ## Equal method
            mi <- minValue(stratImg)
            ma <- maxValue(stratImg)
            groups <- seq(mi, ma, by = (ma - mi)/n)
          }    
          assoc  <- cut(sr$strat, breaks = groups, labels = FALSE, include.lowest = TRUE)
          gMax   <- tail(groups, 1)
          gMin   <- groups[1]
          tab    <- tabulate(assoc, nbins = (length(groups)-1))
          tooFew <- which(tab < minN) + 1
          while(length(tooFew)){
            tooFew[tooFew == 1] <- 2
            tooFew[tooFew == length(groups)] <- length(groups) -1
            groups <- groups[-tooFew]
            groups <- unique(c(gMin, groups, gMax))
            assoc  <- cut(sr$strat, breaks = groups, labels = FALSE, include.lowest = TRUE)
            tab    <- tabulate(assoc, nbins = (length(groups)-1))
            tooFew <- which(tab < minN) + 1
          }           
        }
        #.vMessage("Estimate coefficients")
        x     <- if(method == "stat") sr$illu else log(sr$illu/cos(sz))
        kl <- lapply(1:nlayers(img), function(i){
          if(method == "stat") {
            y <- sr[,i] 
          } else {
            stz <- sr[,i] < 0
            if(any(stz)) {
              warning("Resetting negative reflectances to zero!", call.=FALSE)
              sr[stz,i] <- 1e-32
            } 
            sr[sr[,i] == 0,i] <- 1e-32
            y <- log(sr[,i])
          }
          k <- lapply(1:(length(groups)-1), function(g){
            select <- assoc == g
            mod <- lm(y[select] ~ x[select])
            k <- coefficients(mod)                            
          })
          do.call("rbind", k)
        })
        return(list(groups = groups, k = kl))
      }
      
      
      ###########################################

      #########
      # perform topographic correction
      s2.tc <- topCor_beckschaefer(s2, dgm, solarAngles = c(sun.a, sun.z), method='minnaert', stratImg = 'slope', nStrat=10)
      writeRaster(s2.tc, filename=paste(s2.path,'_TC', '.tif', sep=''), format="GTiff", datatype='INT2S')
      gc()
      ########
      
#################
# TEIL III - calculate vegetation indices
#################
      
      if(calculate_vegetation_indices == TRUE){
        
        s2 <- s2.tc
        rm(s2.tc); gc()
        
        ndvi <- ((s2[[7]]-s2[[3]])/(s2[[7]]+s2[[3]]))*10000
        gli <- ((2*s2[[2]]-s2[[3]]-s2[[1]])/(2*s2[[2]]+s2[[3]]+s2[[1]]))*10000
        pbi <- (s2[[7]]/s2[[2]])*1000
        ngrdi <- ((s2[[2]]-s2[[3]])/(s2[[2]]+s2[[3]]))*10000
        cvi <- (s2[[7]]*(s2[[3]]/s2[[2]]^2))*1000
        gndvi <- ((s2[[7]]-s2[[2]])/(s2[[7]]+s2[[2]]))*10000
        bndvi <- ((s2[[7]]-s2[[1]])/(s2[[7]]+s2[[1]]))*10000
        gc()
        s2.vi <- stack(ndvi, gli, pbi, ngrdi, cvi, gndvi, bndvi)
        gc()
        writeRaster(s2.vi, filename=paste(s2.path,'_TC_VI1', '.tif', sep=''), format="GTiff", datatype='INT2S', overwrite=TRUE)
        rm(ndvi, gli, pbi, ngrdi, cvi, gndvi, bndvi, s2.vi); gc()
        
        mcari <- (((s2[[4]] - s2[[3]]) - (s2[[4]] - s2[[2]])) * (s2[[4]]/s2[[3]]))
        mndwi <- ((s2[[2]]-s2[[10]])/(s2[[2]]+s2[[10]]))*10000
        mtci <- ((s2[[5]]-s2[[4]])/(s2[[4]]+s2[[3]]))*1000
        ndrei1 <- ((s2[[5]]-s2[[4]])/(s2[[5]]+s2[[4]]))*10000
        ndrei2 <- ((s2[[6]]-s2[[4]])/(s2[[6]]+s2[[4]]))*10000
        slavi <- (s2[[7]]/(s2[[3]]+s2[[10]]))*1000
        ndwi1 <- ((s2[[7]]-s2[[9]])/(s2[[7]]+s2[[9]]))*10000
        ndwi2 <- ((s2[[7]]-s2[[10]])/(s2[[7]]+s2[[10]]))*10000
        ireci <- ((s2[[7]]-s2[[3]])/(s2[[4]]+s2[[5]]))*1000
        gc()
        s2.vi <- stack(mcari, mndwi, mtci, ndrei1, ndrei2, slavi, ndwi1, ndwi2, ireci)
        gc()
        writeRaster(s2.vi, filename=paste(s2.path,'_TC_VI2', '.tif', sep=''), format="GTiff", datatype='INT2S', overwrite=TRUE)
        rm(mcari, mndwi, mtci, ndrei1, ndrei2, slavi, ndwi1, ndwi2, ireci); gc()
        }
      
    