# -*- coding: utf-8 -*-
"""
===============================================================================
file            : Metriken_Holzvolumen_F3.py
===============================================================================

This Python script derives predictors for modeling timber volume and above-ground 
biomass from normalized digital surface models (nDSM) in laz-format. The predictors 
comprise the average and maximum height of the nDSM, the standard deviation, 95th 
and the 75th percentile of height in an area of 20 x 20 m. Also the percentage of 
pixels with height > 6 m as well as percentage of pixels with height > 20 m in a 
20 x 20 m area is calculated.

It was devised within the scope of the "F3 - Flächendeckende 
Fernerkundungsbasierte Forstliche Strukturdaten" (F3- Area-wide remote sensing 
based forest structural data) project by project partners Forest Research 
Institute of Baden-Württemberg (Forstliche Versuchs- und Forschungsanstalt 
Baden-Württemberg - FVA) and Northwest German Forest Research Institute 
(Nordwestdeutsche Forstliche Versuchsanstalt - NW-FVA).
For further information go to www.waldwissen.net/technik/inventur/f3/ or contact
Petra Adler, Petra.Adler@forst.bwl.de (FVA)
Jörg Ackermann, Joerg.Ackermann@nw-fva.de (NW-FVA)

This script is published under GNU General Public License Version 3, 29 June 2007.

"""

# ==============================================================================
# Import moduls
# ==============================================================================
import os
import glob
import multiprocessing
from shutil import rmtree
import arcpy


# ==============================================================================
# Define functions
# ==============================================================================

def create_folders(out_folder, metric, pixel_sizes):
    """If not already exists, create folders with the name of the metric and the pixel size."""""
    for pixel in pixel_sizes:
        folder = r"{}\{}_{}".format(out_folder, metric, pixel)
        if not os.path.isdir(folder):
            os.mkdir(folder)
            print("Directory {}\{}_{} created.".format(out_folder, metric, pixel))


def create_lof(input_dir):
    """ Create and return a list of files as .txt from the files in the input directory"""
    dirList = [[os.path.join(root, pointcloud) for pointcloud in files] for root, dirs, files in
               os.walk(os.path.abspath(input_dir))]
    with open('files.txt', 'w') as f:
        [f.write("{}\n".format(item)) for item in dirList[0]]
    f.close()
    return r"files.txt"


def remove_folders(out_f, metric, pixel_sizes):
    """If folder exists, delete it."""""
    for pixel in pixel_sizes:
        folder = r"{}\{}_{}".format(out_f, metric, pixel)
        if os.path.isdir(folder):
            rmtree(folder)
            print("Directory {}\{}_{} removed.".format(out_f, metric, pixel))


def worker():

    output_rau = os.path.join(output_folder, "Metriken_Holzvolumen")
    if not os.path.isdir(output_rau):
        os.mkdir(output_rau)
        print("{} created".format(output_rau))


    #     # Calculate mean of heights
    create_folders(output_rau, 'mean', pixel_step)
    for n in pixel_step:
        parameters = " -drop_withheld -step {} -epsg {} -mean -cores {} " \
                     " -otif -odir {}\mean_{}".format(n, epsg_code, cores, output_rau, n)
        lastools_query = lasgrid + " -lof " + create_lof(cloud_dsm) + parameters
        os.system(lastools_query)

        # change prefix of files names
        current_output_dir = os.path.join(output_rau, "mean_" + str(n))
        for fn in os.listdir(current_output_dir):
            fn_new = "mean" + str(n) + "_" + fn[4:]
            os.rename(os.path.join(current_output_dir, fn), os.path.join(current_output_dir, fn_new))


    #     # Calculate max of heights

    create_folders(output_rau, 'max', pixel_step)
    for n in pixel_step:
        parameters = " -drop_withheld -step {} -epsg {} -max -cores {} " \
                     "-otif -odir {}\max_{}".format(n, epsg_code, cores, output_rau, n)
        lastools_query = lasgrid + " -lof " + create_lof(cloud_dsm) + parameters
        os.system(lastools_query)

        # change prefix of files names
        current_output_dir = os.path.join(output_rau, "max_" + str(n))
        for fn in os.listdir(current_output_dir):
            fn_new = "max" + str(n) + "_" + fn[4:]
            os.rename(os.path.join(current_output_dir, fn), os.path.join(current_output_dir, fn_new))

    #     # Calculate std of heights

    create_folders(output_rau, 'std', pixel_step)
    for n in pixel_step:
        parameters = " -drop_withheld -step {} -epsg {} -std -cores {} " \
                     "-otif -odir {}\std_{}".format(n, epsg_code, cores, output_rau, n)
        lastools_query = lasgrid + " -lof " + create_lof(cloud_dsm) + parameters
        os.system(lastools_query)

        # change prefix of files names
        current_output_dir = os.path.join(output_rau, "std_" + str(n))
        for fn in os.listdir(current_output_dir):
            fn_new = "std" + str(n) + "_" + fn[4:]
            os.rename(os.path.join(current_output_dir, fn), os.path.join(current_output_dir, fn_new))


    #     # Calculate 95-Percentile  of heights

    create_folders(output_rau, 'perc95', pixel_step)
    for n in pixel_step:
        parameters = " -drop_withheld -height_cutoff -1 -drop_withheld -step {} -epsg {} -p 95 -cores {} " \
                     "-otif -odir {}\perc95_{} -odix _{}".format(n, epsg_code, cores, output_rau, n, n)
        lastools_query = lascanopy + " -lof " + create_lof(cloud_dsm) + parameters
        os.system(lastools_query)

        # change prefix of files names
        current_output_dir = os.path.join(output_rau, "perc95_" + str(n))
        for fn in os.listdir(current_output_dir):
            fn_new = "perc95" + str(n) + "_" + fn[4:]
            os.rename(os.path.join(current_output_dir, fn), os.path.join(current_output_dir, fn_new))


    #     # Calculate 75-Percentile  of heights

    create_folders(output_rau, 'perc75', pixel_step)
    for n in pixel_step:
        parameters = " -drop_withheld -height_cutoff -1 -drop_withheld -step {} -epsg {} -p 75 -cores {} " \
                     "-otif -odir {}\perc75_{} -odix _{}".format(n, epsg_code, cores, output_rau, n, n)
        lastools_query = lascanopy + " -lof " + create_lof(cloud_dsm) + parameters
        os.system(lastools_query)

        # change prefix of files names
        current_output_dir = os.path.join(output_rau, "perc75_" + str(n))
        for fn in os.listdir(current_output_dir):
            fn_new = "perc75" + str(n) + "_" + fn[4:]
            os.rename(os.path.join(current_output_dir, fn), os.path.join(current_output_dir, fn_new))


        # Calculate cctot6 (crown cover of trees >6 m)

    create_folders(output_rau, 'cctot6', pixel_step)
    for n in pixel_step:
        parameters = " -drop_withheld -height_cutoff -1 -step {} -epsg {} -d 6 55 -cores {} " \
                     "-otif -odir {}\cctot6_{} -odix _{}".format(n, epsg_code, cores, output_rau, n, n)
        lastools_query = lascanopy + " -lof " + create_lof(cloud_dsm) + parameters
        os.system(lastools_query)

        # change prefix of files names
        current_output_dir = os.path.join(output_rau, "cctot6_" + str(n))
        for fn in os.listdir(current_output_dir):
            fn_new = "cctot6" + str(n) + "_" + fn[4:]
            os.rename(os.path.join(current_output_dir, fn), os.path.join(current_output_dir, fn_new))


    #     # Calculate cctot20 (crown cover of trees >20 m)

    create_folders(output_rau, 'cctot20', pixel_step)
    for n in pixel_step:
        parameters = " -drop_withheld -height_cutoff -1 -step {} -epsg {} -d 20 55 -cores {} " \
                     "-otif -odir {}\cctot20_{} -odix _{}".format(n, epsg_code, cores, output_rau, n, n)
        lastools_query = lascanopy + " -lof " + create_lof(cloud_dsm) + parameters
        os.system(lastools_query)

        # change prefix of files names
        current_output_dir = os.path.join(output_rau, "cctot20_" + str(n))
        for fn in os.listdir(current_output_dir):
            fn_new = "cctot20" + str(n) + "_" + fn[4:]
            os.rename(os.path.join(current_output_dir, fn), os.path.join(current_output_dir, fn_new))

# ==============================================================================
# Start processing
# ==============================================================================
if __name__ == "__main__":
    # ==============================================================================
    # Start: Set paths and define parameters
    # ==============================================================================
    # Paths
    # path to dsm point cloud (ndsm_laz)
    cloud_dsm = r"D:\F3\Daten\Abgeleitete_Daten\Aus_Luftbildern\Oberflaechenmodelle\nDSM_laz"
    # path to output folder
    output_folder = r"D:\F3\Daten\Abgeleitete_Daten\Aus_Luftbildern\Metriken"

    # coordinate reference system
    # define crs that is used (GK2 = 31466, GK3 = 31467, GK4 = 31468, UTM Zone 32N = 25832, UTM Zone 33N = 25833)
    epsg_code = "25832"

    # Define path to bin folder of LAStools, e.g. C:\LAStools\bin
    lastools = r"C:\LAStools\bin"
    # ==============================================================================
    # End: Set paths and define parameters
    # ==============================================================================

    lascanopy = lastools + r"\\lascanopy.exe"
    lasdiff = lastools + r"\\lasdiff.exe"
    lasgrid = lastools + r"\\lasgrid.exe"

    # Parameters
    cores = multiprocessing.cpu_count() - 1  # define parameters for parallel processing
    pixel_step = [20]  # define geometric resolution (in meters) of output data

    letsgo = worker()

