# ===============================================================================
# file            : Baumarten_Sentinel2_F3.R
# ===============================================================================

# This R script produces a tree species map as well as an accuracy matrix that 
# allows calculating the accuracy of the map. 

# It was devised within the scope of the "F3 - Flchendeckende 
# Fernerkundungsbasierte Forstliche Strukturdaten" (F3- Area-wide remote sensing 
# based forest structural data) project by project partners Forest Research 
# Institute of Baden-Wrttemberg (Forstliche Versuchs- und Forschungsanstalt 
# Baden-Wrttemberg - FVA) and Northwest German Forest Research Institute 
# (Nordwestdeutsche Forstliche Versuchsanstalt - NW-FVA).
# For further information go to www.waldwissen.net/technik/inventur/f3/ or contact
# Petra Adler, Petra.Adler@forst.bwl.de (FVA)
# Jrg Ackermann, Joerg.Ackermann@nw-fva.de (NW-FVA)

# This script is published under GNU General Public License Version 3, 29 June 2007.

# =================================================================================


library(raster)
library(maptools)
library(rgdal)
library(randomForest)
library(RStoolbox)
library(plotKML)
library(caret)
library(ggplot2)
library(doSNOW)
library(doParallel)
library(foreach)
library(dplyr)
library(sampling)

#######################################################
# Bitte setzten Sie die korrekten Pfade und spezifizieren 
# Sie die geforderten Parameter!
#######################################################

setwd('D:/F3/Daten/Sentinel_2')

# Anzahl der Prozessoren, die fr die Prozessierung genutzt werden soll.
    nkerne <- 6

# Ausdehnung (Extent) des Gebiets, das kartiert werden soll und auf dessen Flche sich das Accuracy Assessment beziehen soll.
    Gebietsausdehnung <- extent(526000,563000,5716000,5747000) # xmin, xmax, ymin, ymax

# Pfad zu Waldmaske mit den folgenden Werten: 1 = Wald, NA = Nicht-Wald. 
    waldmaske <- raster('D:/F3/Daten/Zusatzdaten/DLM_basis/waldmaske.tif')

# Pfad zu Referenzdaten (Shape-Datei, Punkte).  
    path.shp <- 'D:/F3/Daten/Zusatzdaten/Forst/BI_shape/BI_Baumart.shp'
  
  # Bitte geben Sie an, in welcher Spalte der Attributtabelle sich die Baumarten befinden.
    spalte.baumart <- 'Baumart'

# Pfad zu Sentinel-2 Daten
    path.s <- 'S2A_sen2cor_noTopCor/S2B_MSIL2A_20170823T103019_N0205_R108_T32UNC_20170823T103018_Dsen2_TopCorSlope10K06ndvi.tif'

# Dateiname der Baumartenkarte
    mapname <- 'Baumarten_F3.tif' 

###################################################################
###################################################################        
    path.vi1 <- gsub('.tif', '_VI1.tif', path.s) 
    path.vi2 <- gsub('.tif', '_VI2.tif', path.s)

#Einlesen und zuschneiden der Sentinel-2 Daten
    sat <- stack(path.s, path.vi1, path.vi2)
    sat <- crop(sat, Gebietsausdehnung)

# Einlesen der Refenzdaten.
    vec <- readOGR(path.shp)

# Reprojektion der Referenzdaten zu Projektion der Sentinel-2 Daten.
    vec <- reproject(vec, sat@crs@projargs)

# Resampling der Waldmaske auf Pixelgre der Sentinel-2 Daten
    waldmaske <- resample(waldmaske, sat[[1]])

###################
# Extraktion der Sentinel-2 Pixelwerte an den Positionen der Referenzdatenpunkte.    
    beginCluster(n=nkerne)
    train_raw <- extract(sat, vec, sp=T, df=F)
    endCluster()
#################
# Vorbereiten der Datentabelle. Selektion der Spalten Baumart & der Spalten mit den Pixelwerten 
    dat <- as.data.frame(train_raw)
    dat <- subset(dat, select=-c(coords.x1,coords.x2))

    colnames(dat)[colnames(dat) == spalte.baumart] <- 'Baumart'
    
    datx <- as.data.frame(dat[,c('Baumart')])
    colnames(datx) <- 'Baumart'
    dat <- dat[,c((dim(dat)[2]+1-dim(sat)[3]) : (dim(dat)[2]))]
    dat <- cbind(datx, dat)
    dat <- na.omit(dat) 

    dat$Baumart <- factor(dat$Baumart)
    table(dat$Baumart)
    boxplot(dat[,10] ~ dat$Baumart)

################
# Modellierung mit Random Forest
    rf.ntree <- 500

    # Random Forest Model
    rf.1b <- randomForest(dat$Baumart~., data=dat, ntree=rf.ntree, proximity=F, importance=T, keep.forest=T)
    rf.1b
    varImpPlot(rf.1b, type=1)
    ma <- rf.1b$confusion
    ma <- ma[,-dim(ma)[2]]
    ma <- t(ma)
    dyn <- colnames(ma)

# Anwendung des RF Modells auf die Satellitendaten -> Baumartenkarte
    map <- predict(sat, rf.1b, progress='text', type='response')
    map <- map * waldmaske
    writeRaster(map, filename=mapname, format='GTiff', overwrite=T, datatype='INT1U')

# Resampling der Baumartenkarte auf 20 m Auflsung
    map3.20 <- aggregate(map, fact=2, fun=modal, na.rm=T)
    writeRaster(map3.20, filename=gsub('.tif', '_20m.tif', mapname), format='GTiff', overwrite=T, datatype='INT1U')

##########################
### Accuracy Assessment following Olofsson et al. (2014), Good practices for estimating area and assessing accuracy of land change, 
# Remote Sensing of Environment 148 (2014) 42-57,
# https://www.sciencedirect.com/science/article/abs/pii/S0034425714000704 

# calculate the area proportions for each map class
    pixelsize <- res(map)[1] * res(map)[2] / 10000 # the size of each pixel in ha 
    map.vec <- as.vector(map)
    maparea <- table(map.vec)
    maparea <- as.vector(maparea)
    aoi <- sum(maparea)
    propmaparea <- maparea/aoi

# convert the absolute cross tab into a probability cross tab
    ni. <- rowSums(ma) # number of reference points per map class
    propma <-  as.matrix(ma/ni. * propmaparea)
    propma[is.nan(propma)] <- 0 # for classes with ni. = 0

# estimate the accuracies
    OA <- sum(diag(propma)) # overall accuracy (Eq. 1 in Olofsson et al. 2014)
    UA <- diag(as.matrix(propma)) / rowSums(propma) # user's accuracy (Eq. 2 in Olofsson et al. 2014)
    PA <- diag(as.matrix(propma)) / colSums(propma) # producer's accuracy (Eq. 3 in Olofsson et al. 2014)

# estimate confidence intervals for the accuracies
    V_OA <- sum(propmaparea^2 * UA * (1 - UA) / (ni. - 1), na.rm=T)  # variance of overall accuracy (Eq. 5 in Olofsson et al. 2014)
    V_UA <- UA * (1 - UA) / (rowSums(ma) - 1) # variance of user's accuracy (Eq. 6 in Olofsson et al. 2014)

# variance of producer's accuracy (Eq. 7 in Olofsson et al. 2014)
    N.j <- array(0, dim=length(dyn))
    aftersumsign <- array(0, dim=length(dyn))
    for(cj in 1:length(dyn)) {
      N.j[cj] <- sum(maparea / ni. * ma[, cj], na.rm=T)
      aftersumsign[cj] <- sum(maparea[-cj]^2 * ma[-cj, cj] / ni.[-cj] * ( 1 - ma[-cj, cj] / ni.[-cj]) / (ni.[-cj] - 1), na.rm = T)
        }
    V_PA <- 1/N.j^2 * ( 
      maparea^2 * (1-PA)^2 * UA * (1-UA) / (ni.-1) + 
        PA^2 * aftersumsign
        ) 
    V_PA[is.nan(V_PA)] <- 0

##########################################################################
### Estimate area ########################################################
##########################################################################

# proportional area estimation
    propAreaEst <- colSums(propma) # proportion of area (Eq. 8 in Olofsson et al. 2014)
    AreaEst <- propAreaEst * sum(maparea) # estimated area

# standard errors of the area estimation (Eq. 10 in Olofsson et al. 2014)
    V_propAreaEst <- array(0, dim=length(dyn))
    for (cj in 1:length(dyn)) {
      V_propAreaEst[cj] <- sum((propmaparea * propma[, cj] - propma[, cj] ^ 2) / ( rowSums(ma) - 1), na.rm=T)
        }
    V_propAreaEst[is.na(V_propAreaEst)] <- 0

# produce the overview table
    ov <- as.data.frame(round(propma, 3))*100
    ov$maparea <- maparea * pixelsize # in ha
    ov$prop_maparea <- round(propmaparea, 3)*100
    
    ov$adj_proparea <- round(propAreaEst, 3)*100
    ov$CI_adj_proparea <- round(1.96 * sqrt(V_propAreaEst), 3)*100
    ov$adj_area <- round(ov$adj_proparea * aoi * pixelsize, 3) # in ha
    ov$CI_adj_area <- round(1.96 * sqrt(V_propAreaEst) * aoi * pixelsize, 3) # in ha
    ov$UA <- round(UA, 3)*100
    ov$CI_UA <- round(1.96 * sqrt(V_UA), 3)*100
    ov$PA <- round(PA, 3)*100
    ov$CI_PA <- round(1.96 * sqrt(V_PA), 3)*100
    rownames(ov) <- colnames(ma)
    ov$OA <- c(round(OA, 3), rep(NA, times = length(dyn) - 1))*100
    ov$CI_OA <- c(round(1.96 * sqrt(V_OA), 3), rep(NA, times = length(dyn) - 1))*100
    Baumart <- row.names(ma)
    ov2 <- cbind(ma, Baumart, ov)
    ov2
    write.csv2(ov2, file=gsub('.tif', '_accuracy.csv', mapname))
    
    mp <- barplot(ov2$prop_maparea, ylim=c(0,max(ov2$prop_maparea+10)), cex.names=1.5, names.arg=dyn, ylab='Flchenanteil [%]')
    text(mp, ov2$prop_maparea+2, labels=ov2$prop_maparea, cex=1.5)

